// Copyright (c) 1995 K.Yoshizawa (PAF02413.niftyserve.or.jp)
// This file is partially derived from nt.c which is ...
// Copyright (c) 1990,1991,1992 Chris and John Downey

/***

* program name:
    wvi
* function:
    PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
    win32.cpp
* module function:
    Definitions for Windows NT system interface module.
* history:
    STEVIE - ST Editor for VI Enthusiasts, Version 3.10
    Originally by Tim Thompson (twitch!tjt)
    Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
    Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

//***************************************************************
//																*
// SECTION : EXPORTED VARIABLES									*
//																*
//***************************************************************

// Size of standard console screen (hStdOut)
int iRows, iColumns;

// Standard handles
HANDLE hStdIn, hStdOut, hStdErr;

//***************************************************************
//																*
// SECTION : LOCAL VARIABLES AND MACRO DEFINITIONS				*
//																*
//***************************************************************

static BOOL ConsoleViModeFirstTime = TRUE;

#define PRINTF_BUF_SIZE 4096

//***************************************************************
//																*
// SECTION : API_FATAL ERROR HANDLER FOR API CALLS				*
//																*
//***************************************************************

_declspec(thread) LPSTR lpszFatalErrorFileName;	// This is an always ANSI string.
_declspec(thread) int iFatalErrorLine;

void Win32ApiFatalError(LPWSTR format, ...)
{
	va_list ap;
	LPWSTR p;
	WCHAR szMsg[256];
	DWORD dwWritten;

	wsprintf(szMsg, L"\nFILE:%hs  LINE:%d  API Error Code:%ld\n"
	, lpszFatalErrorFileName, iFatalErrorLine, (long)GetLastError());

	// to find <null> character at the end of 'szMsg'
	for (p = szMsg; *p != L'\0'; p++)
		;

	va_start(ap, format);
	wvsprintf(p, format, ap);
	va_end(ap);

	WriteConsole(hStdErr, szMsg, wcslen(szMsg), &dwWritten, NULL);

	ExitProcess(1);
}

//***************************************************************
//																*
// SECTION : CONSOLE INPUT FUNCTIONS							*
//																*
//***************************************************************

static DWORD dwOldConsoleInputMode;

static void ConsoleInputViMode()
{
	// when turning off ENABLE_LINE_INPUT, you MUST also turn off ENABLE_ECHO_INPUT.
	if ( ! SetConsoleMode(hStdIn,
		(dwOldConsoleInputMode & ~(ENABLE_LINE_INPUT | ENABLE_ECHO_INPUT | ENABLE_PROCESSED_INPUT))
		| ENABLE_WINDOW_INPUT | ENABLE_MOUSE_INPUT)
	) {
		API_FATAL(L"SetConsoleMode");
	}
}

static void ConsoleInputNormalMode()
{
	if ( ! SetConsoleMode(hStdIn, dwOldConsoleInputMode)) {
		API_FATAL(L"SetConsoleMode");
	}
}

static int iVkTable[] = {
	EOF,		// 0x00
	EOF,		// 0x01 VK_LBUTTON
	EOF,		// 0x02 VK_RBUTTON
	EOF,		// 0x03 VK_CANCEL
	EOF,		// 0x04 VK_MBUTTON
	EOF,		// 0x05
	EOF,		// 0x06
	EOF,		// 0x07
	L'\b',		// 0x08 VK_BACK
	L'\t',		// 0x09 VK_TAB
	EOF,		// 0x0a
	EOF,		// 0x0b
	EOF,		// 0x0c VK_CLEAR
	L'\r',		// 0x0d VK_RETURN
	EOF,		// 0x0e
	EOF,		// 0x0f

	EOF,		// 0x10 VK_SHIFT
	EOF,		// 0x11 VK_CONTROL
	EOF,		// 0x12 VK_MENU
	EOF,		// 0x13 VK_PAUSE
	EOF,		// 0x14 VK_CAPITAL
	EOF,		// 0x15
	EOF,		// 0x16
	EOF,		// 0x17
	EOF,		// 0x18
	EOF,		// 0x19
	EOF,		// 0x1a
	L'\033', 	// 0x1b VK_ESCAPE
	EOF,		// 0x1c
	EOF,		// 0x1d
	EOF,		// 0x1e
	EOF,		// 0x1f

	L' ',		// 0x20 VK_SPACE
	K_PGUP,		// 0x21 VK_PRIOR
	K_PGDOWN,	// 0x22 VK_NEXT
	K_END,		// 0x23 VK_END
	K_HOME,		// 0x24 VK_HOME
	K_LARROW,	// 0x25 VK_LEFT
	K_UARROW,	// 0x26 VK_UP
	K_RARROW,	// 0x27 VK_RIGHT
	K_DARROW,	// 0x28 VK_DOWN
	EOF,		// 0x29 VK_SELECT
	EOF,		// 0x2a VK_PRINT
	EOF,		// 0x2b VK_EXECUTE
	EOF,		// 0x2c VK_SNAPSHOT or VK_COPY
	K_INSERT,	// 0x2d VK_INSERT
	K_DELETE,	// 0x2e VK_DELETE
	K_HELP ,	// 0x2f VK_HELP

	EOF,		// 0x30 '0'
	EOF,		// 0x31 '1'
	EOF,		// 0x32 '2'
	EOF,		// 0x33 '3'
	EOF,		// 0x34 '4'
	EOF,		// 0x35 '5'
	EOF,		// 0x36 '6'
	EOF,		// 0x37 '7'
	EOF,		// 0x38 '8'
	EOF,		// 0x39 '9'
	EOF,		// 0x3a
	EOF,		// 0x3b
	EOF,		// 0x3c
	EOF,		// 0x3d
	EOF,		// 0x3e
	EOF,		// 0x3f

	EOF,		// 0x40
	EOF,		// 0x41 'A'
	EOF,		// 0x42 'B'
	EOF,		// 0x43 'C'
	EOF,		// 0x44 'D'
	EOF,		// 0x45 'E'
	EOF,		// 0x46 'F'
	EOF,		// 0x47 'G'
	EOF,		// 0x48 'H'
	EOF,		// 0x49 'I'
	EOF,		// 0x4a 'J'
	EOF,		// 0x4b 'K'
	EOF,		// 0x4c 'L'
	EOF,		// 0x4d 'M'
	EOF,		// 0x4e 'N'
	EOF,		// 0x4f 'O'

	EOF,		// 0x50 'P'
	EOF,		// 0x51 'Q'
	EOF,		// 0x52 'R'
	EOF,		// 0x53 'S'
	EOF,		// 0x54 'T'
	EOF,		// 0x55 'U'
	EOF,		// 0x56 'V'
	EOF,		// 0x57 'W'
	EOF,		// 0x58 'X'
	EOF,		// 0x59 'Y'
	EOF,		// 0x5a 'Z'
	EOF,		// 0x5b
	EOF,		// 0x5c
	EOF,		// 0x5d
	EOF,		// 0x5e
	EOF,		// 0x5f

	L'0',		// 0x60 VK_NUMPAD0
	L'1',		// 0x61 VK_NUMPAD1
	L'2',		// 0x62 VK_NUMPAD2
	L'3',		// 0x63 VK_NUMPAD3
	L'4',		// 0x64 VK_NUMPAD4
	L'5',		// 0x65 VK_NUMPAD5
	L'6',		// 0x66 VK_NUMPAD6
	L'7',		// 0x67 VK_NUMPAD7
	L'8',		// 0x68 VK_NUMPAD8
	L'9',		// 0x69 VK_NUMPAD9
	L'*',		// 0x6a VK_MULTIPLY
	L'+',		// 0x6b VK_ADD
	EOF,		// 0x6c VK_SEPARATOR
	L'-',		// 0x6d VK_SUBTRACT
	L'.',		// 0x6e VK_DECIMAL
	L'/',		// 0x6f VK_DIVIDE

	EOF,		// 0x70 VK_F1
	EOF,		// 0x71 VK_F2
	EOF,		// 0x72 VK_F3
	EOF,		// 0x73 VK_F4
	EOF,		// 0x74 VK_F5
	EOF,		// 0x75 VK_F6
	EOF,		// 0x76 VK_F7
	EOF,		// 0x77 VK_F8
	EOF,		// 0x78 VK_F9
	EOF,		// 0x79 VK_F10
	EOF,		// 0x7a VK_F11
	EOF,		// 0x7b VK_F12
	EOF,		// 0x7c VK_F13
	EOF,		// 0x7d VK_F14
	EOF,		// 0x7e VK_F15
	EOF,		// 0x7f VK_F16

	EOF,		// 0x80 VK_F17
	EOF,		// 0x81 VK_F18
	EOF,		// 0x82 VK_F19
	EOF,		// 0x83 VK_F20
	EOF,		// 0x84 VK_F21
	EOF,		// 0x85 VK_F22
	EOF,		// 0x86 VK_F23
	EOF,		// 0x87 VK_F24
	EOF,		// 0x88
	EOF,		// 0x89
	EOF,		// 0x8a
	EOF,		// 0x8b
	EOF,		// 0x8c
	EOF,		// 0x8d
	EOF,		// 0x8e
	EOF,		// 0x8f

	EOF,		// 0x90 VK_NUMLOCK
	EOF,		// 0x91 VK_SCROLL
	EOF,		// 0x92
	EOF,		// 0x93
	EOF,		// 0x94
	EOF,		// 0x95
	EOF,		// 0x96
	EOF,		// 0x97
	EOF,		// 0x98
	EOF,		// 0x99
	EOF,		// 0x9a
	EOF,		// 0x9b
	EOF,		// 0x9c
	EOF,		// 0x9d
	EOF,		// 0x9e
	EOF,		// 0x9f
};

#define VKTABLESIZE (sizeof(iVkTable) / sizeof(iVkTable[0]))

static int ConvertVK(WORD wCode)
{
	if (wCode >= VKTABLESIZE) return EOF;
	return iVkTable[wCode];
}

int ConsoleGetc(DWORD dwMsecTimeout)
{
	DWORD dwRead;
	DWORD dwEvent;
	INPUT_RECORD ir;
	int iChar;

	for (;;) {
		dwEvent = WaitForSingleObject(hStdIn, dwMsecTimeout==0 ? INFINITE : dwMsecTimeout);
		if (dwEvent == WAIT_FAILED) {
			API_FATAL(L"WaitForSingleObject");
		}
		if (dwEvent == WAIT_TIMEOUT) {
			return EOF;
		}

		if ( ! ReadConsoleInput(hStdIn, &ir, 1, &dwRead)) {
			API_FATAL(L"ReadConsoleInput");
		}

		switch(ir.EventType) {
		case KEY_EVENT:
			if (ir.Event.KeyEvent.bKeyDown) {
				if ((iChar = ConvertVK(ir.Event.KeyEvent.wVirtualKeyCode)) != EOF) {
					return iChar;
				}
				if (ir.Event.KeyEvent.uChar.UnicodeChar) {
					return ir.Event.KeyEvent.uChar.UnicodeChar;
				}
			}
			break;
		case MOUSE_EVENT:
		case WINDOW_BUFFER_SIZE_EVENT:
		case MENU_EVENT:
		case FOCUS_EVENT:
			break;
		}
	}
}

//***************************************************************
//																*
// SECTION : CONSOLE OUTPUT FUNCTIONS							*
//																*
//***************************************************************

static CONSOLE_SCREEN_BUFFER_INFO csbiStdOut;
static SMALL_RECT srctWindowRect;
static COORD coordWindowSize;
static COORD coordCurPos;

#define STDOUTBUFSIZE 256
static WCHAR achStdOutBuf[STDOUTBUFSIZE];
static int iStdOutBufPos;

static void ConsoleOutputViMode()
{
	if ( ! GetConsoleScreenBufferInfo(hStdOut, &csbiStdOut)) {
		API_FATAL(L"GetConsoleScreenBufferInfo");
	}

	iRows = csbiStdOut.dwSize.Y - SYSLINE;
	iColumns = csbiStdOut.dwSize.X;

	srctWindowRect.Left = 0;
	srctWindowRect.Top = 0;
	srctWindowRect.Right = iColumns-1;
	srctWindowRect.Bottom = iRows-1;
	coordWindowSize.X = iColumns;
	coordWindowSize.Y = iRows;

	iStdOutBufPos = 0;
}

static void ConsoleOutputNormalMode()
{
	ConsoleFlushOutput();

	if ( ! SetConsoleTextAttribute(hStdOut, csbiStdOut.wAttributes)) {
		API_FATAL(L"SetConsoleTextAttribute");
	}
}

int ConsolePutc(WCHAR ch, HANDLE hConsole)
{
	DWORD dwWritten;

	if (ch == DUMMY_CHAR) return ch;

	if (hConsole != hStdOut) {
		if ( ! WriteConsole(hConsole, &ch, 1, &dwWritten, NULL)) {
			API_FATAL(L"WriteConsole() in ConsolePutc");
		}
	} else {
		if (iStdOutBufPos >= STDOUTBUFSIZE) ConsoleFlushOutput();
		achStdOutBuf[iStdOutBufPos++] = ch;
	}
	return ch;
}

int ConsolePuts(LPCWSTR lpcsz, HANDLE hConsole)
{
	LPCWSTR p;

	for (p = lpcsz; *p != L'\0'; p++) ConsolePutc(*p, hConsole);
	return p - lpcsz;
}

int ConsoleHprintf(HANDLE hConsole, LPCWSTR lpcszFormat, ...)
{
	va_list ap;
	WCHAR szBuf[PRINTF_BUF_SIZE];
	int n;

	va_start(ap, lpcszFormat);
	n = wvsprintf(szBuf, lpcszFormat, ap);
	va_end(ap);

	ConsolePuts(szBuf, hConsole);

	return n;
}

int ConsolePrintf(LPCWSTR lpcszFormat, ...)
{
	va_list ap;
	WCHAR szBuf[PRINTF_BUF_SIZE];
	int n;

	va_start(ap, lpcszFormat);
	n = wvsprintf(szBuf, lpcszFormat, ap);
	va_end(ap);

	ConsolePuts(szBuf, hStdOut);

	return n;
}

void ConsoleFlushOutput()
{
	DWORD dwWritten;

	if (iStdOutBufPos == 0) return;

	if ( ! WriteConsole(hStdOut, achStdOutBuf, iStdOutBufPos, &dwWritten, NULL)) {
		API_FATAL(L"WriteConsole() in ConsolePutc");
	}

	iStdOutBufPos = 0;
}

void ConsoleEraseScreen()
{
	COORD coord = { 0, 0 };
	DWORD dwWritten;

	ConsoleFlushOutput();

	if ( ! FillConsoleOutputCharacter(
		hStdOut,
		L' ',
		coordWindowSize.X * (coordWindowSize.Y + SYSLINE),
		coord,
		&dwWritten)
	) {
		API_FATAL(L"FillConsoleOutputAttribute");
	}

	if ( ! FillConsoleOutputAttribute(
		hStdOut,
		Pn(P_color),
		coordWindowSize.X * (coordWindowSize.Y + SYSLINE),
		coord,
		&dwWritten)
	) {
		API_FATAL(L"FillConsoleOutputAttribute");
	}
}

void ConsoleEraseLine()
{
	DWORD dwWritten;

	ConsoleFlushOutput();

	if ( ! FillConsoleOutputCharacter(
		hStdOut,
		L' ',
		coordWindowSize.X - coordCurPos.X,
		coordCurPos,
		&dwWritten)
	) {
		API_FATAL(L"FillConsoleOutputAttribute");
	}

	if ( ! FillConsoleOutputAttribute(
		hStdOut,
		Pn(P_color),
		coordWindowSize.X - coordCurPos.X,
		coordCurPos,
		&dwWritten)
	) {
		API_FATAL(L"FillConsoleOutputAttribute");
	}
}

void ConsoleMoveCursor(int iRow, int iCol)
{
	CONSOLE_SCREEN_BUFFER_INFO csbi;

	coordCurPos.X = iCol;
	coordCurPos.Y = iRow;

	if ( ! GetConsoleScreenBufferInfo(hStdOut, &csbi)) {
		API_FATAL(L"GetConsoleScreenBufferInfo");
	}
	if (csbi.dwCursorPosition.X == iCol && csbi.dwCursorPosition.Y == iRow) return;

	ConsoleFlushOutput();

	if ( ! SetConsoleCursorPosition(hStdOut, coordCurPos)) {
		API_FATAL(L"SetConsoleCursorPosition");
	}
}

void ConsoleScrollDown(int iStartRow, int iEndRow, int iLines)
{
	SMALL_RECT srct;
	COORD coordNew;
	CHAR_INFO charinfo;

	ConsoleFlushOutput();

	srct.Top = iStartRow;
	srct.Bottom = iEndRow;
	srct.Left = 0;
	srct.Right = coordWindowSize.X - 1;
	coordNew.X = 0;
	coordNew.Y = iStartRow + iLines;
	charinfo.Char.UnicodeChar = L' ';
	charinfo.Attributes = Pn(P_color);
	if ( ! ScrollConsoleScreenBuffer(hStdOut, &srct, &srct, coordNew, &charinfo)) {
		API_FATAL(L"ScrollConsoleScreenBuffer");
	}
}

void ConsoleScrollUp(int iStartRow, int iEndRow, int iLines)
{
	ConsoleScrollDown(iStartRow, iEndRow, -iLines);
}

//***************************************************************
//																*
// SECTION : CONSOLE COLOR FUNCTIONS							*
//																*
//***************************************************************

#define FOREGROUND_YELLOW   (FOREGROUND_RED|FOREGROUND_GREEN)
#define BACKGROUND_YELLOW   (BACKGROUND_RED|BACKGROUND_GREEN)

void ConsoleAutoSetupColor()
{
	HANDLE hStdOut;
	CONSOLE_SCREEN_BUFFER_INFO csbiStdOut;
	WORD w, wBack;

	if ((hStdOut = GetStdHandle(STD_OUTPUT_HANDLE)) == INVALID_HANDLE_VALUE) {
		API_FATAL(L"GetStdHandle");
	}

	if ( ! GetConsoleScreenBufferInfo(hStdOut, &csbiStdOut)) {
		API_FATAL(L"GetConsoleScreenBufferInfo");
	}

	Pn(P_color) = csbiStdOut.wAttributes;

	w = csbiStdOut.wAttributes
	& (BACKGROUND_RED|BACKGROUND_GREEN|BACKGROUND_BLUE|BACKGROUND_INTENSITY|FOREGROUND_INTENSITY);

	wBack = csbiStdOut.wAttributes & (BACKGROUND_RED|BACKGROUND_GREEN|BACKGROUND_BLUE);

	Pn(P_roscolor)    = (wBack!=BACKGROUND_YELLOW ? (w|FOREGROUND_YELLOW) : csbiStdOut.wAttributes);
	Pn(P_statuscolor) = (wBack!=BACKGROUND_GREEN ? (w|FOREGROUND_GREEN) : csbiStdOut.wAttributes);
	Pn(P_systemcolor) = csbiStdOut.wAttributes;
}

void ConsoleSelectColor(int n)
{
	ConsoleFlushOutput();
	if ( ! SetConsoleTextAttribute(hStdOut, n)) {
		API_FATAL(L"SetConsoleTextAttribute");
	}
}

//***************************************************************
//																*
// SECTION : CONSOLE INITIALIZATION								*
//																*
//***************************************************************

void ConsoleViMode()
{
	if (ConsoleViModeFirstTime) {
		// Get standard Handles

		hStdOut = GetStdHandle(STD_OUTPUT_HANDLE);
		if(hStdOut == INVALID_HANDLE_VALUE) API_FATAL(L"GetStdHandle");

		hStdErr = GetStdHandle(STD_ERROR_HANDLE);
		if(hStdErr == INVALID_HANDLE_VALUE) API_FATAL(L"GetStdHandle");

		hStdIn = GetStdHandle(STD_INPUT_HANDLE);
		if(hStdIn == INVALID_HANDLE_VALUE) API_FATAL(L"GetStdHandle");

		GetConsoleMode(hStdIn, &dwOldConsoleInputMode);
		if(hStdIn == INVALID_HANDLE_VALUE) API_FATAL(L"GetConsoleMode");
	}
	ConsoleOutputViMode();
	ConsoleInputViMode();
	// ConsoleSelectColor(Pn(P_color));
}

void ConsoleNormalMode()
{
	ConsoleOutputNormalMode();
	ConsoleInputNormalMode();
}

//***************************************************************
//																*
// SECTION : FILE TEST											*
//																*
//***************************************************************

BOOL FileExists(LPCWSTR lpcszFileName)
{
	return GetFileAttributes(lpcszFileName) != 0xffffffff;
}

BOOL CanWriteOnFile(LPCWSTR lpcszFileName)
{
	DWORD dw;

	dw = GetFileAttributes(lpcszFileName);
	return
		dw == 0xffffffff	// The file does not exist. Ok, we can write on it.
	||
		( ! (dw & FILE_ATTRIBUTE_READONLY)		// The file is NOT read-only.
		&& ! (dw & FILE_ATTRIBUTE_DIRECTORY)	// The file is NOT directory.
		)
	;
}

//***************************************************************
//																*
// SECTION : FILE OPEN/CLOSE									*
//																*
//***************************************************************

LPFILESTREAM FileOpen(LPCWSTR lpcszFileName, LPCWSTR lpcszMode, CODETYPE ct)
{
	FILE		*fp;
	WORD		wCurrentMagic;
	WORD		wUnicodeMagic = 0xfeff;
	BOOL		bUnicodeMagicExists = FALSE;
	WCHAR		mode[256];	// maybe eough size
	WCHAR		*p;
	BOOL 		bSeekToEndOfFile = FALSE;
	BOOL		fWriteMode = FALSE;
	LPFILESTREAM lpfs;

	// test the file has UNICODE magic number if the file exists
	if ((fp = _wfopen(lpcszFileName, L"rb")) != NULL) {
		if (fread(&wCurrentMagic, 2, 1, fp) == 1
		&&  wCurrentMagic == wUnicodeMagic) {
			bUnicodeMagicExists = TRUE;
		}
		fclose(fp);
	}

	// file mode option
	p = mode;
	for (; *lpcszMode != L'\0'; lpcszMode++) {
		switch (*lpcszMode) {
		case L'a':
			bSeekToEndOfFile = TRUE;
			// no break
		case L'w':
			fWriteMode = TRUE;
			// no break
		case L'r':
		// case L'+': not yet ...
			*p++ = *lpcszMode;
			break;
		default:
			return NULL;
		}
	}

	*p++ = L'b';		// open as binary file always
	*p++ = L'\0';

	if ((fp = _wfopen(lpcszFileName, mode)) == NULL) return NULL;

	if (fWriteMode) {
		if (ct == CODETYPE_UNICODE) {
			fseek(fp, 0, SEEK_SET);
			if (fwrite(&wUnicodeMagic, 2, 1, fp) != 1) {
				fclose(fp);
				return NULL;
			}
		}
	} else { // read only
		if (bUnicodeMagicExists) {
			fseek(fp, 2, SEEK_SET);
			ct = CODETYPE_UNICODE;
		} else {
			if (ct == CODETYPE_UNICODE) ct = CODETYPE_MULTIBYTE;
		}
	}

	if (bSeekToEndOfFile) {
		fseek(fp, 0, SEEK_END);
	}

	if ((lpfs = (LPFILESTREAM)malloc(sizeof(FILESTREAM))) == NULL) {
		fclose(fp);
		return NULL;
	}
	setvbuf(fp, (char*)lpfs->achBuf, _IOFBF, FILESTREAMBUFSIZE);
	lpfs->fp = fp;
	lpfs->ct = ct;
	lpfs->iChar = EOF;
	lpfs->iPreFetched = EOF;

	return lpfs;
}

int FileClose(LPFILESTREAM lpfs)
{
	int n;

	n = fclose(lpfs->fp);
	free(lpfs);
	return n;
}

//***************************************************************
//																*
// SECTION : FILE INPUT											*
//																*
//***************************************************************

int FileGetc(LPFILESTREAM lpfs)
{
	int c;
	BYTE byBuf[2];
	WCHAR wcBuf[2];
	int nLen;

	if (lpfs->iChar != EOF) {
		c = lpfs->iChar;
		lpfs->iChar = EOF;
		return c;
	}

	switch (lpfs->ct) {
	case CODETYPE_BINARY:
		if (fread(&byBuf[0], 1, 1, lpfs->fp) == 0) return EOF;
		return byBuf[0];
	case CODETYPE_UNICODE:
		if (fread(&wcBuf[0], 2, 1, lpfs->fp) == 0) return EOF;
		return wcBuf[0];
	case CODETYPE_MULTIBYTE:
		// Because "API IsDBCSLeadByte()" doesn't work for arbitrary code pages,
		// we can't avoid using an ugly technique. Sorry ...
		if (lpfs->iPreFetched != EOF) {
			byBuf[0] = lpfs->iPreFetched;
			lpfs->iPreFetched = EOF;
			nLen = 1;
			nLen += fread(&byBuf[1], 1, 1, lpfs->fp);
		} else {
			if ((nLen = fread(&byBuf[0], 1, 2, lpfs->fp)) == 0) return EOF;
		}
		if (MultiByteToWideChar(Pn(P_codepage), MB_PRECOMPOSED, (LPSTR)byBuf, nLen
		, (LPWSTR)&wcBuf, 2) == 2) {
			// Two bytes are translated into two wide characters.
			// OK, we assume byBuf[0] is NOT a lead byte of DBCS.
			lpfs->iPreFetched = byBuf[1];	// Save the second byte.
		}
		return wcBuf[0];
	}
	return EOF; // dummy return
}

int FileUngetc(int c, LPFILESTREAM lpfs)
{
	lpfs->iChar = c;
	return c;
}

LPWSTR FileGets(LPWSTR lpszBuf, int iBufSize, LPFILESTREAM lpfs)
{
	LPWSTR p = lpszBuf;
	int c;

	if (iBufSize <= 0) return NULL;

	while (iBufSize-- >= 2) {
		c = FileGetc(lpfs);
		if (c == EOF) break;
		*lpszBuf++ = c;
		if (c == L'\n') break;
	}
	*lpszBuf = L'\0';
	if (p == lpszBuf) return NULL;
	return lpszBuf;
}

//***************************************************************
//																*
// SECTION : FILE OUTPUT										*
//																*
//***************************************************************

int FilePutc(WCHAR wc, LPFILESTREAM lpfs)
{
	BYTE byBuf[16];
	int nLen;

	switch (lpfs->ct) {
	case CODETYPE_BINARY:
		if (fputc(wc, lpfs->fp) == EOF) return EOF;
		break;
	case CODETYPE_UNICODE:
		if (fwrite(&wc, 2, 1, lpfs->fp) != 1) return EOF;
		break;
	case CODETYPE_MULTIBYTE:
		nLen = WideCharToMultiByte(Pn(P_codepage), WC_COMPOSITECHECK, &wc, 1
			, (LPSTR)byBuf, sizeof byBuf, NULL, NULL);
		if (fwrite(byBuf, 1, nLen, lpfs->fp) != nLen) return EOF;
		break;
	}
	return wc;
}

int FilePuts(LPCWSTR lpcsz, LPFILESTREAM lpfs)
{
	LPCWSTR p;

	for (p = lpcsz; *p != L'\0'; p++) {
		if (FilePutc(*p, lpfs) == EOF) return EOF;
	}
	return p - lpcsz;
}

int FilePrintf(LPFILESTREAM lpfs, LPCWSTR lpcszFormat, ...)
{
	va_list ap;
	WCHAR szBuf[PRINTF_BUF_SIZE];
	int n;

	va_start(ap, lpcszFormat);
	n = wvsprintf(szBuf, lpcszFormat, ap);
	va_end(ap);

	FilePuts(szBuf, lpfs);

	return n;
}

//***************************************************************
//																*
// SECTION : FILE UTILITY										*
//																*
//***************************************************************

// Construct unique name for temporary file, to be used as a backup or a temporary
// file for the named file.
LPWSTR TempFileName(LPCWSTR lpcszOriginalFileName)
{
	int		i, j;
	WCHAR	szHomeDrive[MAXENVLEN];
	WCHAR	szHomeDir[MAXENVLEN];
	LPCWSTR	lpcszOrigBase;
	LPWSTR	lpszTempFileName;
	LPWSTR	lpszSuffix;
	size_t	iSizeOfTempFileName;

	if ((lpcszOrigBase = wcsrchr(lpcszOriginalFileName, L'\\')) == NULL
	&&  (lpcszOrigBase = wcschr(lpcszOriginalFileName, L':')) == NULL
	) {
		lpcszOrigBase = lpcszOriginalFileName;
	} else {
		lpcszOrigBase++;
	}

	if ( ! GetEnvironmentVariable(L"HOMEDRIVE", szHomeDrive, sizeof(szHomeDrive)/sizeof(WCHAR))
	||   ! GetEnvironmentVariable(L"HOMEPATH",  szHomeDir,   sizeof(szHomeDir)/sizeof(WCHAR))) {
		szHomeDrive[0] = L'\0';
		szHomeDir[0] = L'\0';
	} else {
		wcscat(szHomeDir, L"\\");
	}

	// +4 for L".???", +1 for L'\0'
	iSizeOfTempFileName
		= wcslen(szHomeDrive)
		+ wcslen(szHomeDir)
		+ wcslen(lpcszOrigBase)
		+ 4
		+ 1;
	lpszTempFileName = (LPWSTR)calloc(iSizeOfTempFileName, sizeof(WCHAR));
	wcscpy(lpszTempFileName, szHomeDrive);
	wcscat(lpszTempFileName, szHomeDir);
	wcscat(lpszTempFileName, lpcszOrigBase);

	// replace all L'/' by L'\\'
	for (i=0; lpszTempFileName[i] != L'\0'; i++) {
		if (lpszTempFileName[i] == L'/') lpszTempFileName[i] = L'\\';
	}

	// get position to where we add suffix.
	lpszSuffix = &lpszTempFileName[i];
	*lpszSuffix++ = L'.';

	// The suffix ".$$$" is commonly used for temporary file names on
	// MS-DOS & OS/2 systems. We also use the sequence ".$$1", ".$$2" ...
	// ".fff" (all digits are hexadecimal).
	//
	// I assume (and hope) that ".fff" is large enough. (K.Y.)
	for (i=0; i<=0xfff; i++) {
		wsprintf(lpszSuffix, L"%03x", i);
		for (j=0; j<3; j++) {
			if (lpszSuffix[j] != L'0') break;
			lpszSuffix[j] = '$';
		}
		if ( ! FileExists(lpszTempFileName)) break;
	}
	return lpszTempFileName;
}

//***************************************************************
//																*
// SECTION : SIGNAL												*
//																*
//***************************************************************

static BOOL myHandler(DWORD dwCtrlType)
{
	switch(dwCtrlType){

	case CTRL_C_EVENT:
		// used to be: kbdintr = 1;
		do_preserve();
		ExitProcess(0);
		return TRUE;

	case CTRL_BREAK_EVENT:
	case CTRL_CLOSE_EVENT:
		do_preserve();
		ExitProcess(0);
		return TRUE;
	}
	return FALSE;
}

void IgnoreSignals()
{
	SetConsoleCtrlHandler((PHANDLER_ROUTINE) myHandler, FALSE);
}

void CatchSignals()
{
	SetConsoleCtrlHandler((PHANDLER_ROUTINE) myHandler, TRUE);
}

//***************************************************************
//																*
// SECTION : PROCESS CONTROL									*
//																*
//***************************************************************

void SystemExit(int iReturnCode)
{
	ConsoleMoveCursor(iRows - 1, 0);
	ConsoleNormalMode();
	ExitProcess(iReturnCode);
}

DWORD CallSystem(LPWSTR lpszCmdLine)
{
	STARTUPINFO si;
	PROCESS_INFORMATION pi;
	SECURITY_ATTRIBUTES sa;
	DWORD dwExitCode;

	// prepare security attribute
	memset(&sa, 0, sizeof(sa));
	sa.nLength = sizeof(sa);
	sa.bInheritHandle = TRUE;	// Child processes may access handles

	// process start up information
	memset(&si, 0, sizeof(si));
	si.cb = sizeof(si);
    si.wShowWindow = SW_SHOWDEFAULT;
	si.dwFlags = STARTF_USESTDHANDLES;
	si.hStdInput = hStdIn;
	si.hStdOutput = hStdOut;
	si.hStdError = hStdOut;

	// Run the command
	if (CreateProcess(
		NULL,
		lpszCmdLine,
		&sa,
		&sa,
		TRUE,
		0,
		NULL,
		NULL,
		&si,
		&pi)
	) {
		// Success !
		CloseHandle(pi.hThread);
		WaitForSingleObject(pi.hProcess, INFINITE);
		GetExitCodeProcess(pi.hProcess, &dwExitCode);
		dwExitCode = 0;	// Ignore 'REAL' exit code. (0 means 'no error')
		CloseHandle(pi.hProcess);
	} else {
		dwExitCode = 1;
	}

	return dwExitCode;
}

// Fake out a pipe by writing output to temp file, running a process with
// i/o redirected from this file to another temp file, and then reading
// the second temp file back in.
BOOL SystemPipe(LPCWSTR lpcszCmd, int (*WriteFunc)(LPFILESTREAM), int (*ReadFunc)(LPFILESTREAM))
{
	BOOL			bRetval = FALSE;
	LPWSTR			lpszTemp1 = NULL;
	LPWSTR			lpszTemp2 = NULL;
	LPFILESTREAM	lpfsFile1;
	LPFILESTREAM	lpfsFile2;
	HANDLE			hFile1;
	HANDLE			hFile2;
	STARTUPINFO 	si;
	PROCESS_INFORMATION pi;
	WCHAR			szCmdLine[4096];
	SECURITY_ATTRIBUTES sa;

	// Create first temporary file ...
	if ((lpszTemp1 = TempFileName(L"xvi_out")) == NULL
	||  (lpfsFile1 = FileOpen(lpszTemp1, L"w", PIPE_TCT)) == NULL) goto end;

	// ... then write to it & close it ...
	if ((*WriteFunc)(lpfsFile1) == EOF) {
		FileClose(lpfsFile1);
		goto end;
	}
	if (FileClose(lpfsFile1) != 0) goto end;

	// prepare security attribute
	memset(&sa, 0, sizeof(sa));
	sa.nLength = sizeof(sa);
	sa.bInheritHandle = TRUE;	// Child processes may access handles

	// ... then re-open it for reading
	hFile1 = CreateFile(
		lpszTemp1,
		GENERIC_READ,
		FILE_SHARE_READ,
		&sa,
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL,
		NULL);

	if (hFile1 == INVALID_HANDLE_VALUE) goto end;

	// ... then open second one for writing
	if ((lpszTemp2 = TempFileName(L"xvi_out")) == NULL) goto end;

	hFile2 = CreateFile(
		lpszTemp2,
		GENERIC_WRITE,
		FILE_SHARE_READ,
		&sa,
		CREATE_ALWAYS,
		FILE_ATTRIBUTE_NORMAL,
		NULL);

	if (hFile2 == INVALID_HANDLE_VALUE) goto end;

	// Run the command
	wsprintf(szCmdLine, L"cmd.exe /C %s", lpcszCmd);

	memset(&si, 0, sizeof(si));
	si.cb = sizeof(si);
    si.wShowWindow = SW_SHOWDEFAULT;
	si.dwFlags = STARTF_USESTDHANDLES;
	si.hStdInput = hFile1;
	si.hStdOutput = hFile2;
	si.hStdError = hFile2;

	if (CreateProcess(
		NULL,
		szCmdLine,
		NULL,
		NULL,
		TRUE,
		0,
		NULL,
		NULL,
		&si,
		&pi)
	) {
		// Success !
		CloseHandle(pi.hThread);
		WaitForSingleObject(pi.hProcess, INFINITE);
		CloseHandle(pi.hProcess);
	} else {
		// Fail !
		CloseHandle(hFile1);
		CloseHandle(hFile2);
		goto end;
	}

	if ( ! CloseHandle(hFile1) || ! CloseHandle(hFile2)) goto end;

	// Now read from the second temporary file and close it.
	if ((lpfsFile2 = FileOpen(lpszTemp2, L"r", PIPE_TCT)) == NULL) goto end;
	if ((*ReadFunc)(lpfsFile2) == EOF) {
		FileClose(lpfsFile2);
		goto end;
	}
	if (FileClose(lpfsFile2) != 0) goto end;

	bRetval = TRUE;

end:
	// Clean up.
	if (lpszTemp1 != NULL) {
		DeleteFile(lpszTemp1);
		free(lpszTemp1);
	}
	if (lpszTemp2 != NULL) {
		DeleteFile(lpszTemp2);
		free(lpszTemp2);
	}

	return bRetval;
}

//***************************************************************
//																*
// SECTION : MISC.												*
//																*
//***************************************************************

void AlertBell()
{
	Beep(600,100);
}

// Check and set value of P_codepage parameter.
BOOL set_codepage(Xviwin *window, Paramval new_value, BOOL interactive)
{
	if ( ! IsValidCodePage(new_value.pv_i)) {
		if (interactive) {
			show_error(window, L"Invalid code page (%d)", new_value.pv_i);
		}
		return FALSE;
	}
	return TRUE;
}
