// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)startup.c	2.3 (Chris & John Downey) 9/4/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	main.c
* module function:
	Entry point for xvi; setup, argument parsing and signal handling.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

// References to the current cursor position, and the window
// and buffer into which it references. These make the code a
// lot simpler, but we have to be a bit careful to update them
// whenever necessary.
Buffer	*curbuf;
Xviwin	*curwin;

// Global variables.
state_t 		State = NORMAL; // This is the current state of the command 
								// interpreter. 

unsigned		echo;			// bitmap controlling the verbosity of
								// screen output (see xvi.h for details).

int 			indentchars;	// number of chars indented on current line 

volatile WCHAR
				kbdintr;		// global flag set when a keyboard interrupt
								// is received

BOOL			imessage;		// global flag to indicate whether we should
								// display the "Interrupted" message

// Internal routines.
static	void	usage(void);

Xviwin *xvi_startup(VirtScr *vs, int argc, WCHAR *argv[])
{
	WCHAR		*tag = NULL;			// tag from command line 
	WCHAR		*pat = NULL;			// pattern from command line 
	long		line = -1;				// line number from command line 
	WCHAR		**files;
	int 		numfiles = 0;
	int 		count;

	IgnoreSignals();

	// Initialise vischar() function in unicode module.
	init_vischar();

	// Initialise parameter module.
	init_params();

	// Initialise yank/put module.
	init_yankput();

	// The critical path this code has to follow is quite tricky.
	// We can't really run the "env" string until after we've set
	// up the first screen window because we don't know what the
	// commands in "env" might do: they might, for all we know,
	// want to display something. And we can't set up the first
	// screen window until we've set up the terminal interface.
	//
	// Also, we can't read the command line arguments until after
	// we've run the "env" string because a "-s param=value" argument
	// should override any setting of that parameter in the environment.
	//
	// All this means that the usage() function, which tells the
	// user that the command line syntax was wrong, can only be
	// called after the display interface has already been set up,
	// which means we must be in visual mode. So usage() has to
	// switch back to system mode (otherwise, on systems where
	// ConsoleViMode() & ConsoleNormalMode() {ex- sys_startv() & sys_endv()}
	// switch display pages, the user will never see the output of
	// usage()). So ...

	// Set up the first buffer and screen window.
	// Must call sys_init first.
	curbuf = new_buffer();
	if (curbuf == NULL) {
		ConsoleNormalMode();
		ConsolePuts(L"Can't allocate buffer memory.\n", hStdErr);
		SystemExit(2);
	}
	curwin = init_window(vs);
	if (curwin == NULL) {
		ConsoleNormalMode();
		ConsolePuts(L"Can't allocate buffer memory.\n", hStdErr);
		SystemExit(2);
	}

	// Connect the two together.
	map_window_onto_buffer(curwin, curbuf);

	init_sline(curwin);

	// Try to obtain a value for the "shell" parameter from the
	// environment variable SHELL. If this is NULL, do not override
	// any existing value. The system interface code (sys_init()) is
	// free to set up a default value, and the initialisation string
	// in the next part of the startup is free to override both that
	// value and the one from the environment.
	{
		WCHAR	sh[MAXENVLEN];

		if (GetEnv(L"ComSpec", sh, MAXENVLEN) != 0) {
			set_param(P_shell, sh);
		}
	}

	// Get module file name, make HELP file name, and regist it.
	{
		WCHAR	helpfile[MAX_PATH];

		if (GetModuleFileName(NULL, helpfile, MAX_PATH) != 0) {
			wcscpy(wcsrchr(helpfile, L'.'), L".hlp");
			set_param(P_helpfile, helpfile);
		}
	}

	// Get environment variable 'WVIINIT' and parse it.
	{
		WCHAR	env[MAXENVLEN];

		if (GetEnv(L"WVIINIT", env, MAXENVLEN) != 0) {
			parse_param(curwin, env, TRUE);
		}
	}

	// Process the command line arguments.
	//
	// We can't really do this until we have run the "env" string,
	// because "-s param=value" on the command line should override
	// parameter setting in the environment.
	//
	// This is a bit awkward because it means usage() is called
	// when we're already in vi mode (which means, among other
	// things, that display pages may have been swapped).
	for (count = 1;
		count < argc && (argv[count][0] == L'-' || argv[count][0] == L'+');
																count++) {

		if (argv[count][0] == L'-') {
			switch (argv[count][1]) {
			case L't':
				// -t tag or -ttag
				if (numfiles != 0)
					usage();
				if (argv[count][2] != L'\0') {
					tag = &(argv[count][2]);
				} else if (count < (argc - 1)) {
					count += 1;
					tag = argv[count];
				} else {
					usage();
				}
				break;

			case L's':
				// -s param=value or
				// -sparam=value
				if (argv[count][2] != L'\0') {
					argv[count] += 2;
				} else if (count < (argc - 1)) {
					count += 1;
				} else {
					usage();
				}
				do_set(curwin, 1, &argv[count], FALSE);
				break;

			default:
				usage();
			}

		} else { // argv[count][0] == L'+'
			WCHAR		nc;

			// "+n file" or "+/pat file"
			if (count >= (argc - 1))
				usage();

			nc = argv[count][1];
			if (nc == L'/') {
				pat = &(argv[count][2]);
			} else if (iswdigit(nc)) {
				line = _wtol(&(argv[count][1]));
			} else if (nc == L'\0') {
				line = 0;
			} else {
				usage();
			}
			count += 1;
			files = &argv[count];
			numfiles = 1;
		}
	}
	if (numfiles != 0 || tag != NULL) {
		// If we found "-t tag", "+n file" or "+/pat file" on
		// the command line, we don't want to see any more
		// file names.
		if (count < argc)
			usage();
	} else {
		// Otherwise, file names are valid.
		numfiles = argc - count;
		if (numfiles > 0) {
			files = &(argv[count]);
		}
	}

	// Initialise the cursor and top of screen pointers
	// to the start of the first buffer. Note that the
	// bottom of screen pointer is also set up, as some
	// code (e.g. move_window_to_cursor) depends on it.
	curwin->w_topline = curbuf->b_file;
	curwin->w_botline = curbuf->b_file->l_next;
	move_cursor(curwin, curbuf->b_file, 0);
	curwin->w_col = 0;
	curwin->w_row = 0;

	// Clear the window.
	//
	// It doesn't make sense to do this until we have a value for
	// Pn(P_color).
	clear(curwin);

	if (numfiles != 0) {
		if (line < 0 && pat == NULL)
			echo = e_CHARUPDATE | e_SHOWINFO;

		do_next(curwin, numfiles, files, FALSE);

		if (pat != NULL) {
			echo = e_CHARUPDATE | e_SHOWINFO | e_REGERR | e_NOMATCH;
			(void) dosearch(curwin, pat, L'/');
		} else if (line >= 0) {
			echo = e_CHARUPDATE | e_SHOWINFO;
			do_goto((line > 0) ? line : MAX_LINENO);
		}

	} else if (tag != NULL) {
		echo = e_CHARUPDATE | e_SHOWINFO | e_REGERR | e_NOMATCH;
		if (do_tag(curwin, tag, FALSE, TRUE, FALSE) == FALSE) {
			// Failed to find tag - wait for a while
			// to allow user to read tags error and then
			// display the "no file" message.
			Sleep(2000);	// 2000msec. = 2sec.
			show_file_info(curwin);
		}
	} else {
		echo = e_CHARUPDATE | e_SHOWINFO;
		show_file_info(curwin);
	}

	setpcmark(curwin);

	echo = e_CHARUPDATE;

	// Ensure we are at the right screen position.
	move_window_to_cursor(curwin);

	// Draw the screen.
	update_all();

	// Update the cursor position on the screen, and go there.
	cursupdate(curwin);
	wind_goto(curwin);

	// Allow everything.
	echo = e_ANY;

	CatchSignals();

	return(curwin);
}

// Print usage message and die.
//
// This function is only called after we have set the terminal to vi
// mode.
//
// The system interface functions have to ensure that it's safe to
// call sys_exit() when sys_endv() has already been called (& there
// hasn't necessarily been any intervening sys_startv()).
static void usage(void)
{
	ConsoleNormalMode();
	ConsolePuts(L"Usage: wvi { options } [ file ... ]\n");
	ConsolePuts(L"		 wvi { options } -t tag\n");
	ConsolePuts(L"		 wvi { options } +[num] file\n");
	ConsolePuts(L"		 wvi { options } +/pat	file\n");
	ConsolePuts(L"\nOptions are:\n");
	ConsolePuts(L"		 -s [no]boolean-parameter\n");
	ConsolePuts(L"		 -s parameter=value\n");
	SystemExit(1);
}
