// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)preserve.c	2.1 (Chris & John Downey) 7/29/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	preserve.c
* module function:
	Buffer preservation routines.

	The do_preserve() routine saves the contents of all modified
	buffers in temporary files. It can be invoked with the
	:preserve command or it may be called by one of the system
	interface modules when at least PSVKEYS keystrokes have been
	read, & at least Pn(P_preservetime) seconds have elsapsed
	since the last keystroke. (PSVKEYS is defined in xvi.h.) The
	preservebuf() routine can be used to preserve a single buffer.

* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)
***/

#include "xvi.h"

// Names of values for the P_preserve enumerated parameter.
WCHAR	*psv_strings[] =
{
	L"unsafe",
	L"standard",
	L"safe",
	L"paranoid",
	NULL
};

// Open temporary file for given buffer.
static LPFILESTREAM psvfile(Xviwin *window)
{
	Buffer 			*buffer;
	LPFILESTREAM	lpfs;

	buffer = window->w_buffer;

	if (buffer->b_tempfname == NULL) {
		WCHAR	*fname;

		fname = buffer->b_filename;
		if (fname == NULL)
			fname = L"unnamed";
		buffer->b_tempfname = TempFileName(fname);
		if (buffer->b_tempfname == NULL) {
			show_error(window, L"Can't create name for preserve file");
			return(NULL);
		}
	}
	lpfs = FileOpen(buffer->b_tempfname, L"w", PRESERVE_TCT);
	if (lpfs == NULL) {
		show_error(window, L"Can't open preserve file %s",
											buffer->b_tempfname);
	}
	return lpfs;
}

// Write contents of buffer to file & close file. Return TRUE if no
// errors detected.
static BOOL putbuf(Xviwin *wp,  LPFILESTREAM fs)
{
	unsigned long		l1, l2;

	if (put_file(wp, fs, (Line *) NULL, (Line *) NULL, &l1, &l2, SYSTEM_TFF) == FALSE) {
		show_error(wp, L"Error writing preserve file %s",
											wp->w_buffer->b_tempfname);
		return(FALSE);
	} else {
		return(TRUE);
	}
}

// Preserve contents of a single buffer, so that a backup copy is
// available in case something goes wrong while the file itself is
// being written.
//
// This is controlled by the P_preserve parameter: if it's set to
// psv_UNSAFE, we just return. If it's psv_STANDARD, to save time, we
// only preserve the buffer if it doesn't appear to have been
// preserved recently: otherwise, if it's psv_SAFE or psv_PARANOID, we
// always preserve it.
//
// Return FALSE if an error occurs during preservation, otherwise TRUE.
BOOL preservebuf(Xviwin *window)
{
	LPFILESTREAM	lpfs;
	Buffer			*bp;

	if (
		Pn(P_preserve) == psv_UNSAFE
		||
		(
			Pn(P_preserve) == psv_STANDARD
			&&
			// If there is a preserve file already ...
			(bp = window->w_buffer)->b_tempfname != NULL
			&&
			FileExists(bp->b_tempfname)
			&&
			// & a preserve appears to have been done recently ...
			keystrokes < PSVKEYS
		)
	) {
		// ... don't bother.
		return(TRUE);
	}

	lpfs = psvfile(window);
	if (lpfs == NULL) {
		return(FALSE);
	}

	return(putbuf(window, lpfs));
}

// Preserve contents of all modified buffers.
BOOL do_preserve(void)
{
	Xviwin	*wp;
	BOOL	psvstatus = TRUE;

	wp = curwin;
	do {
		if (is_modified(wp->w_buffer)) {
			LPFILESTREAM	lpfs;

			lpfs = psvfile(wp);
			if (lpfs != NULL) {
				if (!putbuf(wp, lpfs))
					psvstatus = FALSE;
			} else {
				psvstatus = FALSE;
			}
		}
	} while ((wp = next_window(wp)) != curwin);

	return(psvstatus);
}
