// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)pipe.c		2.1 (Chris & John Downey) 7/29/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	pipe.c
* module function:
	Handle pipe operators.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

static	int p_write(LPFILESTREAM);
static	int	p_read(LPFILESTREAM);

static	Xviwin	*specwin;
static	Line	*line1, *line2;
static	Line	*newlines;

// This function is called when the ! is typed in initially,
// to specify the range; do_pipe() is then called later on
// when the line has been typed in completely.
//
// Note that we record the first and last+1th lines to make the loop easier.
void specify_pipe_range(Xviwin *window, Line *l1, Line *l2)
{
	// Ensure that the lines specified are in the right order.
	if (l1 != NULL && l2 != NULL && earlier(l2, l1)) {
		Line	*tmp;

		tmp = l1;
		l1 = l2;
		l2 = tmp;
	}
	line1 = (l1 != NULL) ? l1 : window->w_buffer->b_file;
	line2 = (l2 != NULL) ? l2->l_next : window->w_buffer->b_lastline;
	specwin = window;
}

// Pipe the given sequence of lines through the command,
// replacing the old set with its output.
void do_pipe(Xviwin *window, WCHAR *command)
{
	if (line1 == NULL || line2 == NULL || specwin != window) {
		show_error(window,
			L"Internal error: pipe through badly-specified range.");
		return;
	}

	newlines = NULL;
	if (SystemPipe(command, p_write, p_read) && newlines != NULL) {
		repllines(window, line1, cntllines(line1, line2) - 1, newlines);
		update_buffer(window->w_buffer);
		begin_line(window, TRUE);
	} else {
		show_error(window, L"Failed to execute \"%s\"", command);
		redraw_screen();
	}
	cursupdate(window);
}

// Returns the number of lines written, or EOF for failure.
static int p_write(LPFILESTREAM lpfs)
{
	Line		*l;
	long		n;

	for (l = line1, n = 0; l != line2; l = l->l_next, n++) {
		if (FilePuts(l->l_text, lpfs) == EOF) return EOF;
		if (FilePutc(system_eol1(), lpfs) == EOF) return EOF;
		if (system_eol2() != EOF) {
			if (FilePutc(system_eol2(), lpfs) == EOF) return EOF;
		}
	}
	return n;
}

// Returns the number of lines read, or EOF for failure.
static int p_read(LPFILESTREAM lpfs)
{
	Line		*lptr = NULL;	// pointer to list of lines 
	Line		*last = NULL;	// last complete line read in 
	Line		*lp;			// line currently being read in 
	enum {
		at_soln,
		in_line,
		at_eoln,
		at_eof
	}	state;
	WCHAR		*buff;			// tmporary line buffer;
								// text of line
								// being read in
	unsigned 	buffsize;		// current size of 'buff'
#	define	BUFFSIZE_UNIT 1024	// tmporary line buffer is
								// alloated as this size, and
								// lengthened by the size.
	int 		col;			// current column in line 
	int			nlines; 		// number of lines read 

	// Allocate temporary line buffer.
	buffsize = BUFFSIZE_UNIT;
	buff = alloc_wchar(buffsize);
	if (buff == NULL) {
		return EOF;
	}

	col = 0;
	nlines = 0;
	state = at_soln;
	while (state != at_eof) {

		int	c;

		c = FileGetc(lpfs);

		// Nulls are special; they can't show up in the file.
		if (c == L'\0') {
			continue;
		}

		if (c == EOF) {
			if (state != at_soln) {
				// Reached EOF in the middle of a line; what
				// we do here is to pretend we got a properly
				// terminated line, and assume that a
				// subsequent getc will still return EOF.
				state = at_eoln;
			} else {
				state = at_eof;
			}
		} else if (c == system_eol1()) {
			state = at_eoln;
		} else if (c == system_eol2()) {
			continue; // Ignore second EOL character.
		}

		if (col >= buffsize - 1) {
			// Line buffer for current line overflows. We adujust the size of it.
			buffsize += BUFFSIZE_UNIT;
			buff = (WCHAR*)realloc(buff, ((unsigned) buffsize) * sizeof(WCHAR));
			// If this fails, we squeak at the user and
			// then throw away the lines read in so far.
			if (buff == NULL) {
				if (lptr != NULL)
					throw_line(lptr);
				return EOF;
			}
		}

		switch (state) {
	//	case at_eof:
	//			break;

		case at_soln:
		case in_line:
			state = in_line;
			buff[col++] = c;
			break;

		case at_eoln:
			// First null-terminate the current line.
			buff[col] = L'\0';

			// We're just read a line, and
			// we've got at least one character,
			// so we have to allocate a new Line
			// structure.
			//
			// If we can't do it, we throw away
			// the lines we've read in so far, &
			// return gf_NOMEM.
			if ((lp = newline(col + 1)) == NULL) {
				if (lptr != NULL) {
					throw_line(lptr);
				}
				free(buff);
				return EOF;
			}

			// We move data from current line buffer into Line structure.
			wcscpy(lp->l_text, buff);

			// Tack the line onto the end of the list,
			// and then point "last" at it.
			if (lptr == NULL) {
				lptr = lp;
				last = lptr;
			} else {
				last->l_next = lp;
				lp->l_prev = last;
				last = lp;
			}

			nlines++;
			col = 0;
			state = at_soln;
			break;
		}
	}

	newlines = lptr;

	free(buff);

	return nlines;
}
