// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)flexbuf.c	2.2 (Chris & John Downey) 8/27/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	flexbuf.c
* module function:
	Routines for Flexbufs (variable-length FIFO queues).

	Some of the access routines are implemented as macros in xvi.h.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

#define FLEXEXTRA		64

// Append a single character to a Flexbuf. Return FALSE if we've run
// out of space.
//
// Note that the f->fxb_chars array is not necessarily null-terminated.
BOOL flexaddch(Flexbuf *f, int ch)
{
	if (flexempty(f))
		f->fxb_rcnt = f->fxb_wcnt = 0;
	if (f->fxb_wcnt >= f->fxb_max) {
		if (f->fxb_max == 0) {
			if ((f->fxb_chars = alloc_wchar(FLEXEXTRA)) == NULL) {
				return FALSE;
			} else {
				f->fxb_max = FLEXEXTRA;
			}
		} else {
			unsigned newsize = f->fxb_wcnt + FLEXEXTRA;

			if ((f->fxb_chars = (WCHAR*)realloc(f->fxb_chars, newsize * sizeof(WCHAR))) == NULL) {
				f->fxb_wcnt = f->fxb_max = 0;
				return FALSE;
			} else {
				f->fxb_max = newsize;
			}
		}
	}
	f->fxb_chars[f->fxb_wcnt++] = ch;
	return TRUE;
}

// Return contents of a Flexbuf as a null-terminated string.
WCHAR *flexgetstr(Flexbuf *f)
{
	if (!flexempty(f) && flexaddch(f, L'\0')) {
		--f->fxb_wcnt;
		return &f->fxb_chars[f->fxb_rcnt];
	} else {
		return L"";
	}
}

// Remove the first character from a Flexbuf and return it.
int flexpopch(Flexbuf *f)
{
	return flexempty(f) ?
		0 : (WCHAR) f->fxb_chars[f->fxb_rcnt++];
}

// Free storage belonging to a Flexbuf.
void flexdelete(Flexbuf *f)
{
	if (f->fxb_max > 0) {
		(void) free(f->fxb_chars);
		f->fxb_wcnt = f->fxb_max = 0;
	}
}

// The following routines provide for appending formatted data to a
// Flexbuf using a subset of the format specifiers accepted by
// printf(3). The specifiers we understand are currently
//
//		%c %d %ld %lu %s %u
//
// Field width, precision & left justification can also be specified
// as for printf().
//
// The main advantage of this is that we don't have to worry about the
// end of the destination array being overwritten, as we do with
// sprintf(3).

static	unsigned	width;
static	unsigned	prec;
static	BOOL		ljust;

// Append a string to a Flexbuf, truncating the string & adding filler
// characters as specified by the width, prec & ljust variables.
//
// The precision specifier gives the maximum field width.
static BOOL strformat(Flexbuf *f, WCHAR *p)
{
	int c;

	if (width != 0 && !ljust) {
		unsigned len;

		len = wcslen(p);
		if (prec != 0 && prec < len)
			len = prec;
		while (width > len) {
			if (!flexaddch(f, L' '))
				return FALSE;
			--width;
		}
	}

	while ((c = *p++) != L'\0') {
		if (!flexaddch(f, c))
			return FALSE;
		if (width != 0)
			--width;
		if (prec != 0) {
			if (--prec == 0) {
				break;
			}
		}
	}
	while (width != 0) {
		if (!flexaddch(f, L' '))
			return FALSE;
		--width;
	}
	return TRUE;
}

// Given a binary long integer, convert it to a decimal number &
// append it to the end of a Flexbuf.
//
// The precision specifier gives the minimum number of decimal digits.
static BOOL numformat(Flexbuf *f, long n, BOOL uflag)
{
	WCHAR *s;
	unsigned len;

	if (n == 0) {
		// Special case.
		s = L"0";
		len = 1;
	} else {
		const DSTR_CHARS = sizeof (long) * 3 + 2;

		static WCHAR dstr[DSTR_CHARS];
		unsigned long un;
		int neg;

		* (s = &dstr[DSTR_CHARS - 1]) = L'\0';

		if (!uflag && n < 0) {
			neg = 1;
			un = -n;
		} else {
			neg = 0;
			un = n;
		}

		while (un > 0 && s > &dstr[1]) {
			*--s = (WCHAR)((un % 10) + L'0');
			un /= 10;
		}

		if (neg)
			*--s = L'-';
		len = &dstr[DSTR_CHARS - 1] - s;
	}

	while (width > len && width > prec) {
		if (!flexaddch(f, L' '))
			return FALSE;
		--width;
	}

	while (prec > len) {
		if (!flexaddch(f, L'0'))
			return FALSE;
		--prec;
		if (width > 0)
			--width;
	}
	prec = 0;
	return strformat(f, s);
}

// Main formatting routine.
BOOL vformat(Flexbuf *f,  WCHAR *format,  va_list argp)
{
	int c;

	while ((c = *format++) != L'\0') {
		if (c == L'%') {
			static WCHAR cstr[2];
			BOOL lnflag;
			BOOL zflag;

			lnflag = FALSE;
			ljust = FALSE;
			width = 0;
			prec = 0;
			zflag = FALSE;
			if ((c = *format++) == L'-') {
				ljust = TRUE;
				c = *format++;
			}

			// Get width specifier.
			if (c == L'0') {
				if (ljust)
					// It doesn't make sense to
					// have a left-justified
					// numeric field with zero
					// padding.
					return FALSE;
				zflag = TRUE;
				c = *format++;
			}

			while (c && iswdigit(c)) {
				if (width != 0)
					width *= 10;
				width += (c - L'0');
				c = *format++;
			}

			if (c == L'.') {
				// Get precision specifier.
				while ((c = *format++) != L'\0' && iswdigit(c)) {
					if (prec != 0)
						prec *= 10;
					prec += (c - L'0');
				}
			}

			switch (c) {
			case L'%':
				cstr[0] = c;
				if (!strformat(f, cstr))
					return FALSE;
				continue;

			case L'c':
				cstr[0] = va_arg(argp, int);
				if (!strformat(f, cstr))
					return FALSE;
				continue;

			case L'l':
				switch (c = *format++) {
				case L'd':
				case L'u':
					lnflag = TRUE;
					break;
				default:
					// Syntax error.
					return FALSE;
				}
				// fall through ... 

			case L'd':
			case L'u':
			{
				long n;

				if (lnflag)
				{
					n = (c == L'u' ?
								(long) va_arg(argp, unsigned long) :
								va_arg(argp, long));
				} else {
					n = (c == L'u' ?
								(long) va_arg(argp, unsigned int) :
								(long) va_arg(argp, int));
				}

				// For integers, the precision
				// specifier gives the minimum number
				// of decimal digits.
				if (zflag && prec < width) {
					prec = width;
					width = 0;
				}
				if (!numformat(f, n, (c == L'u')))
					return FALSE;
				continue;
			}

			case L's':
			{
				WCHAR *sp;

				if ((sp = va_arg(argp, WCHAR *)) == NULL ||
												!strformat(f, sp)) {
					return FALSE;
				}
				continue;
			}

			default:
				// Syntax error.
				return FALSE;
			}
		} else if (!flexaddch(f, c)) {
			return FALSE;
		}
	}
	return TRUE;
}

// Front end callable with a variable number of arguments.
BOOL lformat(Flexbuf *f, WCHAR *format, ...)
{
	va_list argp;
	BOOL retval;

	va_start(argp, format);
	retval = vformat(f, format, argp);
	va_end(argp);
	return retval;
}
