// Copyright (c) 1990,1991,1992 Chris and John Downey 

/***

* @(#)xvi.h 	2.5 (Chris & John Downey) 9/1/92

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	xvi.h
* module function:
	General definitions for xvi.

	This file should really be split up into several files
	rather than being concentrated into one huge monolith.

* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

//**************************************************************
//															   *
// SECTION 1: ENVIRONMENT									   *
//															   *
// Most of this section is concerned with including the right  *
// header files; we also define some things by hand if the	   *
// appropriate definitions are not provided by the system.	   *
//															   *
//**************************************************************

// for UNICODE & WIN32 compilation

#define WIN32
#define UNICODE
#define _UNICODE

// ANSI/ISO-C system include files ...
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include <signal.h>

// Win32 system include files ...
#include <windows.h>
#include <windowsx.h>
#include <tchar.h>

//**************************************************************
//															   *
// SECTION 2: FUNDAMENTAL TYPES 							   *
//															   *
// These types are used by other included header files. 	   *
// Now, this section is empty ...							   *
//															   *
//**************************************************************


//**************************************************************
//															   *
// SECTION 3: FUNDAMENTAL HEADER FILES						   *
//															   *
// These header files define types which are used by Xvi.	   *
//															   *
//**************************************************************

#include "virtscr.h"

//**************************************************************
//															   *
// SECTION 4: MISCELLANEOUS DEFINITIONS 					   *
//															   *
// Definitions of limits and suchlike used within the editor.  *
//															   *
//**************************************************************

// Minimum number of rows a window can have, including status line.
#define MINROWS 		2

// SLOP is the amount of extra space we get for text on a line during
// editing operations that need more space. This keeps us from calling
// malloc every time we get a character during insert mode. No extra
// space is allocated when the file is initially read.
#define SLOP			10

// The number of characters taken up by the line number
// when "number" is set; up to 6 digits plus two spaces.
#define NUM_SIZE		8
#define NUM_FMT 		L"%6ld  "

// Maximum value for the tabstop parameter.
#define MAX_TABSTOP 	32

// Default timeout for keystrokes (in milliseconds).
// The timeout parameter is set to this value.
#define DEF_TIMEOUT 	200


//**************************************************************
//															   *
// SECTION 5: PARAMETER TYPE DEFINITIONS					   *
//															   *
// These are definitions of types for particular parameters.   *
//															   *
//**************************************************************

// Regular expression search modes - used in search.c.
// These are the integer values to which the P_regextype
// enumerated parameter may be set. Note that these values
// are ordered; e.g. rt_GREP is considered to be earlier
// than, and/or less than, rt_EGREP. If the types are added
// to, this ordering must be maintained. Also note that the
// names in the rt_strings table must follow the same order.
#define rt_TAGS 		0		// only ^ and $ are significant 
#define rt_GREP 		1		// like grep, but with \< and \> 
#define rt_EGREP		2		// like egrep, but with \< and \> 

// Array of names for the P_regextype enumeration, defined in
// search.c.
extern	WCHAR			*rt_strings[];

// Integer values for the P_preserve enumerated parameter. Note that
// the names in psv_strings must follow the same order.
#define psv_UNSAFE		0		// never preserve buffer before writing 
#define psv_STANDARD	1		//
								// only preserve buffer before writing
								// if it hasn't been preserved recently
#define psv_SAFE		2		// always preserve buffer before writing 
#define psv_PARANOID	3		//
								// like psv_SAFE, but never remove the
								// preserve file

// Array of names for the P_preserve enumeration, defined in
// search.c.
extern	WCHAR			*psv_strings[];

// Integer values for the P_format enumerated parameter. These are for
// the formats we know about so far. Note that the entries in
// fmt_strings & tftable (defined in fileio.c) must follow the same order.
enum FILEFORMAT
{
	fmt_CSTRING,
	fmt_MACINTOSH,
	fmt_DOS,
	fmt_QNX,
	fmt_TOS,
	fmt_UNIX,
};

// Array of names for the P_format enumeration.
extern	WCHAR			*fmt_strings[];

// Current P_format name.
extern WCHAR *fmtname;

// Integer values for the P_codetype enumerated parameter. These are for
// the code types we know about so far. Note that the entries in
// codetype_strings (defined in fileio.c) must follow the same order.
enum CODETYPE
{
	CODETYPE_MULTIBYTE,
	CODETYPE_UNICODE,
	CODETYPE_BINARY,
};

// Array of names for the P_codetype enumeration.
extern	WCHAR			*codetype_strings[];

// Current P_codetype name.
extern WCHAR *ctname;

// Integer values for the P_jumpscroll enumerated parameter. Note that
// the entries in js_strings (defined in param.c) must follow the same
// order.
#define js_OFF			0
#define js_AUTO 		1
#define js_ON			2

//**************************************************************
//															   *
// SECTION 6: EDITOR TYPE DEFINITIONS						   *
//															   *
//**************************************************************

// Possible editor states.
enum state_t
{
	NORMAL, 					// command mode 
	INSERT, 					// insert mode 
	REPLACE,					// overwrite mode 
	CMDLINE,					// on the : command line 
	DISPLAY 					// in display mode (e.g. g//p or :set all) 
};

extern	state_t 		State;	// defined in main.c 

// Possible return values for cmd_input(), which deals with command
// line input. This is for commands starting with :, /, ? and !.
enum Cmd_State
{
	cmd_COMPLETE,				// user hit return (cmd line available) 
	cmd_INCOMPLETE, 			// not finished typing command line yet 
	cmd_CANCEL					// user aborted command line 
};

// Possible directions for searching, and opening lines.
#define FORWARD 		0
#define BACKWARD		1

// Line structure and its friends.
//
// The structure used to hold a line; this is used to form
// a doubly-linked list of the lines comprising a file.
//
// The definition for MAX_LINENO depends on the type of
// l_number, and on the size of the machine; we are fairly
// safe setting all bits here so long as l_number is always
// an unsigned type. Should really use a lineno_t here, and
// get rid of the need for a maximum lineno.
struct Line
{
	Line 				*l_prev;		// previous line 
	Line 				*l_next;		// next line 
	WCHAR				*l_text;		// text for this line 
	int 				l_size; 		// actual size of space at 's' 
	unsigned long		l_number;		// line "number" 
};

#define MAX_LINENO		ULONG_MAX

// These pseudo-functions operate on lines in a buffer, returning TRUE
// or FALSE according to whether l1 is later or earlier than l2.
// Note that there is no macro for "same", which is excluded by both
// "earlier" and "later".
#define later(l1, l2)	((l1)->l_number > (l2)->l_number)
#define earlier(l1, l2) ((l1)->l_number < (l2)->l_number)

// This macro gives the line number of line 'l' in buffer 'b'.
#define lineno(b, l)	((l)->l_number)

// Easy ways of finding out whether a given line is the first
// or last line of a buffer, without needing a buffer pointer.
#define is_lastline(lp) ((lp)->l_number == MAX_LINENO)
#define is_line0(lp)	((lp)->l_number == 0)

// Structure used to hold a position in a file;
// this is just a pointer to the line, and an index.
struct Posn
{
	Line				*p_line;		// line we're referencing 
	int 				p_index;		// position within that line 
};

// This is stuff to do with marks - it should be privately defined
// in mark.c, but it needs to be related to each individual buffer.
#define NMARKS	10				// max. # of marks that can be saved 

struct Mark
{
	WCHAR				m_name;
	Posn				m_pos;
};

// Structure used to record a single change to a buffer.
// A change may either be a number of lines replaced with a
// new set, a number of characters removed or a number of
// characters replaced with a new set. Character changes
// never straddle line boundaries. A list of these
// structures forms a complex change. There is also a fourth
// type of "change", which does not actually change the
// buffer, but is simply a record of the cursor position at
// the time of the start of the change. This is needed so
// that the cursor returns to the correct position after an
// "undo".
//
// This entire structure is only used in undo.c and alloc.c, and
// no other code should touch it.
enum change_type
{
	C_LINE,
	C_CHAR,
	C_DEL_CHAR,
	C_POSITION
};

struct Change
{
	Change				*c_next;
	change_type			c_type;
	unsigned long		c_lineno;
	union {
		struct {
			long	cul_nlines;
			Line	*cul_lines;
		}		cu_l;
		struct {
			int 	cuc_index;
			int 	cuc_nchars;
			WCHAR	*cuc_chars;
		}		cu_c;
		struct {
			long	cup_line;
			int 	cup_index;
		}		cu_p;
	}					c_u;
};

#define c_nlines		c_u.cu_l.cul_nlines
#define c_lines 		c_u.cu_l.cul_lines
#define c_index 		c_u.cu_c.cuc_index
#define c_nchars		c_u.cu_c.cuc_nchars
#define c_chars 		c_u.cu_c.cuc_chars
#define c_pline 		c_u.cu_p.cup_line
#define c_pindex		c_u.cu_p.cup_index

// Variable-length FIFO queue of characters.
struct Flexbuf
{
		WCHAR			*fxb_chars; 	// pointer to allocated space 
		unsigned		fxb_max;		// size of allocated space 
		unsigned		fxb_rcnt;		// number of characters read 
		unsigned		fxb_wcnt;		// number of characters written 
// public: 
// Initialize a Flexbuf.
#define 				flexnew(f)		((f)->fxb_wcnt = (f)->fxb_max = 0)
// Reset a Flexbuf by clearing its contents, but without freeing the
// dynamically allocated space.
#define 				flexclear(f)	((f)->fxb_wcnt = 0)
// Test whether a Flexbuf is empty.
#define 				flexempty(f)	((f)->fxb_rcnt >= (f)->fxb_wcnt)
// Remove last character from a Flexbuf.
#define 				flexrmchar(f)	(!flexempty(f) && --(f)->fxb_wcnt)
// Return number of characters in a Flexbuf.
#define 				flexlen(f)		(flexempty(f) ? 0 : \
										(f)->fxb_wcnt - (f)->fxb_rcnt)
};

// Structure used to hold all information about a "buffer" -
// i.e. the representation of a file in memory.
struct Buffer
{
	Line				*b_line0;		// ptr to zeroth line of file 
	Line				*b_file;		// ptr to first line of file 
	Line				*b_lastline;	// ptr to (n+1)th line of file 

	// All of these are allocated, and should be freed
	// before assigning any new value to them.
	WCHAR				*b_filename;	// file name, if any 
	WCHAR				*b_tempfname;	// name for temporary copy of file 

	unsigned int		b_flags;		// flags 

	int 				b_nwindows; 	// no of windows open on this buffer 

	// The following only used in mark.c.
	Mark				b_mlist[NMARKS];		// current marks 
	Mark				b_pcmark;				// previous context mark 
	BOOL				b_pcvalid;				// true if pcmark is valid 

	// The following only used in undo.c.
	unsigned int		b_nlevels;		// number of brackets surrounding 
										// current change to buffer 
	Change				*b_change;		// ptr to list of changes made 

};

// Definitions for the "flags" field of a buffer.
#define FL_MODIFIED 	0x1
#define FL_READONLY 	0x2
#define FL_NOEDIT		0x4
#define is_modified(b)	((b)->b_flags & FL_MODIFIED)
#define is_readonly(b)	(Pb(P_readonly) || ((b)->b_flags & FL_READONLY))
#define not_editable(b) ((b)->b_flags & FL_NOEDIT)

// Structure used to hold information about a "window" -
// this is intimately associated with the Buffer structure.
struct Xviwin
{
	Posn				*w_cursor;		// cursor's position in buffer 

	Buffer				*w_buffer;		// buffer we are a window into 

	Line				*w_topline; 	// line at top of screen 
	Line				*w_botline; 	// line below bottom of screen 

	VirtScr 			*w_vs;			// virtual screen for window 

	unsigned			w_nrows;		// number of rows in window 
	unsigned			w_ncols;		// number of columns in window 
	unsigned			w_winpos;		// row of top line of window 
	unsigned			w_cmdline;		// row of window command line 

	// These are used by the ^O command to store the previous
	// size of the window so that we can return to it.
	int 				w_2winpos;		// last row of top line of window 
	int 				w_2nrows;		// last no of rows in buffer window 
	int 				w_2cmdline; 	// last row of window command line 


	// Allocated within screen.c.
	Flexbuf 			w_statusline;	// status information on status line 


	// These elements are related to the cursor's position in the window.
	int 				w_row, w_col;	// cursor's position in window 

	int 				w_virtcol;		// column number of the file's actual 
										// line, as opposed to the column 
										// number we're at on the screen. 
										// This makes a difference on lines 
										// which span more than one screen 
										// line. 

	int 				w_curswant; 	// The column we'd like to be at. 
										// This is used to try to stay in 
										// the same column through up/down 
										// cursor motions. 

	BOOL				w_set_want_col; // If set, then update w_curswant 
										// the next time through cursupdate() 
										// to the current virtual column 

	int 				w_c_line_size;	// current size of cursor line 

	BOOL				w_curs_new; 	// true if cursor should be updated 

	// The following only used in windows.c.
	Xviwin				*w_last;		// first and last pointers 
	Xviwin				*w_next;
};

// Values returned by inc() & dec().
enum mvtype
{
	mv_NOMOVE = -1, 	// at beginning or end of buffer 
	mv_SAMELINE = 0,	// still within same line 
	mv_CHLINE = 1,		// changed to different line 
	mv_EOL = 2			// in same line, at terminating L'\0' 
};

// Number of input characters since the last buffer preservation.
extern volatile int 	keystrokes;

// Minimum number of keystrokes after which we do an automatic
// preservation of all modified buffers.
#define PSVKEYS 		60

// Exceptional return values for get_file().
#define gf_NEWFILE		((long)-1)		// no such file 
#define gf_CANTOPEN 	((long)-2)		// error opening file 
#define gf_IOERR		((long)-3)		// error reading from file 
#define gf_NOMEM		((long)-4)		// not enough memory 

// Editor input events. Handled by xvi_handle_event().
enum Ev_type
{
	Ev_char,
	Ev_timeout
};

struct xvEvent
{
	Ev_type				ev_type;
	union {
		// Ev_char: 
		int 	evu_inchar;

		// Ev_timeout: 
	}					ev_u;
} ;

#define ev_inchar		ev_u.evu_inchar


//**************************************************************
//															   *
// SECTION 7: MISCELLANEOUS MACROS							   *
//															   *
//**************************************************************


//**************************************************************
//															   *
// SECTION 8: XVI-LOCAL HEADER FILES						   *
//															   *
// Various subsidiary header files with definitions relating   *
// to particular areas of the editor (or its environment).	   *
// Note that these header files may use definitions in this    *
// file, so are included after all types are defined.		   *
//															   *
//**************************************************************

#include "unicode.h"
#include "param.h"
#include "ptrfunc.h"
#include "win32.h"


//**************************************************************
//															   *
// SECTION 9: SYSTEM-SPECIFIC HEADER FILES					   *
//                                                             *
// Now, this section is empty because the taget OS is Win32	   *
// only.                                                       *
//															   *
//**************************************************************

//**************************************************************
//															   *
// SECTION 10: GLOBAL VARIABLE DECLARATIONS 				   *
//															   *
//**************************************************************

// Miscellaneous global vars.
extern	Buffer			*curbuf;		// current buffer 
extern	Xviwin			*curwin;		// current window 

extern	int 			indentchars;	// auto-indentation on current line 
extern	WCHAR			*altfilename;	// name of current alternate file 
extern	WCHAR			Version[];		// version string for :ve command 

// This flag is set when a keyboard interrupt is received.
extern volatile WCHAR kbdintr;

// This one indicates whether we should display the "Interrupted"
// message.
extern	BOOL			imessage;

// This variable (defined in main.c) is a bitmap which controls the
// verbosity of screen output. The meanings of the individual bits
// are:
//
//		e_CHARUPDATE means it's OK to update individual characters in
//		any window.
//
//		e_SCROLL means it's OK to scroll any area of the screen up or
//		down.
//
//		e_REPORT means it's OK for report() to report the number of
//		lines inserted or deleted.
//
//		e_SHOWINFO means it's OK for show_file_info() to display file
//		information for any buffer.
//
//		e_BEEP: not implemented yet.
//
//		e_REGERR means it's OK for functions in search.c to display
//		messages which may have resulted from an invalid regular
//		expression string.
//
//		e_NOMATCH means it's OK for functions in search.c to complain
//		if they fail to match a regular expression at least once.
//
// If we're reading an input sequence & e_CHARUPDATE & e_SCROLL are
// turned off, no screen updating will be done until an ESC is
// received.
extern	unsigned		echo;

#define e_CHARUPDATE	1
#define e_SCROLL		2
#define e_REPORT		4
#define e_SHOWINFO		010
#define e_BEEP			020
#define e_REGERR		040
#define e_NOMATCH		0100

#define e_ANY			0xffff


//**************************************************************
//															   *
// SECTION 11: FUNCTION DECLARATIONS						   *
//															   *
//**************************************************************

// Declarations of all the routines exported from the various "*.cpp" files.

// defscr.cpp 

// alloc.cpp 
void chfree(Change *ch);
Change *challoc(void);
char *alloc_byte(unsigned size);
WCHAR *alloc_wchar(unsigned size);
WCHAR *strsave(const WCHAR *string);
Line *newline(int nchars);
BOOL bufempty(Buffer *b);
BOOL buf1line(Buffer *b);
BOOL endofline(Posn *p);
BOOL grow_line(Line *lp,  int n);
void throw_line(Line *lineptr);

// unicode.cpp 
void init_vischar();
unsigned vischar(int c, WCHAR **pp, int tabcol);

// buffers.cpp 
Buffer *new_buffer(void);
void free_buffer(Buffer *buffer);
BOOL clear_buffer(Buffer *buffer);
extern	int 	nbuffers;

// cmdline.cpp 
void do_colon(WCHAR *cmdline, BOOL interactive);

// cursor.cpp 
void cursupdate(Xviwin *win);
void curs_horiz(Xviwin *win, int nchars);

// edit.cpp 
BOOL i_proc(int c);
void startinsert(int startln);
BOOL r_proc(int c);
void startreplace(int c);
WCHAR *mkstr(int c);

// ex_cmds1.cpp 
void do_quit(Xviwin *window, BOOL force);
void do_split_window(Xviwin *window);
BOOL do_buffer(Xviwin *window, WCHAR *filename);
void do_close_window(Xviwin *win, BOOL force);
void do_xit(Xviwin *window);
BOOL do_edit(Xviwin *window, BOOL force, WCHAR *arg);
void do_args(Xviwin *window);
void do_next(Xviwin *window, int argc, WCHAR *argv[], BOOL force);
void do_rewind(Xviwin *window, BOOL force);
BOOL do_write(Xviwin *window, WCHAR *filename, Line *l1, Line *l2, BOOL force);
void do_wq(Xviwin *window, WCHAR *filename, BOOL force);
void do_read(Xviwin *window, WCHAR *filename, Line *atline);
void do_alt_edit(Xviwin *window);
void do_compare(void);

// ex_cmds2.cpp 
void do_shcmd(Xviwin *window, WCHAR *command);
void do_shell(Xviwin *window);
void do_suspend(Xviwin *window);
void do_equals(Xviwin *window, Line *line);
void do_help(Xviwin *window);
BOOL do_source(BOOL interactive, WCHAR *file);
WCHAR *do_chdir(WCHAR *dir);
void do_cdmy(int type, Line *l1, Line *l2, Line *destline);

// events.cpp 
long xvi_handle_event(xvEvent *ev);

// fileio.cpp 
BOOL set_format(Xviwin *window, Paramval new_value, BOOL interactive);
BOOL set_codetype(Xviwin *window, Paramval new_value, BOOL interactive);
long get_file(Xviwin *window, WCHAR *filename, Line **headp, Line **tailp, WCHAR *extra_str, WCHAR *no_file_str, BOOL fChangeFileInfo);
BOOL write_it(Xviwin *window, WCHAR *fname, Line *start, Line *end, BOOL force);
BOOL put_file(Xviwin *window,  LPFILESTREAM fs, Line *start, Line *end, unsigned long *ncp, unsigned long *nlp, int fmt);
int system_eol1(void);
int system_eol2(void);

// find.cpp 
Posn *searchc(int ch, int dir, int type, int num);
Posn *crepsearch(Buffer *buffer, int flag, int num);
Posn *showmatch(void);
Posn *find_pattern(WCHAR *str, int dir, int num);
Posn *fwd_word(Posn *p, int type, BOOL skip_white);
Posn *bck_word(Posn *p, int type, BOOL skip_white);
Posn *end_word(Posn *p, int type, BOOL skip_white);
BOOL dosearch(Xviwin *window, WCHAR *str, int cmd_char);

// flexbuf.cpp 
BOOL flexaddch( Flexbuf *f, int ch);
WCHAR *flexgetstr(Flexbuf *f);
int flexpopch(Flexbuf *f);
void flexdelete(Flexbuf *f);
BOOL vformat(Flexbuf *f,  WCHAR *format,  va_list argp);
BOOL lformat(Flexbuf *f, WCHAR *format, ...);

// map.cpp 
void stuff(WCHAR *format, ...);
int map_getc(void);
void map_char( int c);
void map_timeout(void);
BOOL map_waiting(void);
void xvi_keymap(WCHAR *lhs, WCHAR *rhs);
void xvi_map(int argc, WCHAR *argv[], BOOL exclam, BOOL inter);
void xvi_unmap(int argc, WCHAR *argv[], BOOL exclam, BOOL inter);

// mark.cpp 
void init_marks(Buffer *buffer);
BOOL setmark(int c, Buffer *buffer, Posn *pos);
void setpcmark(Xviwin *window);
Posn *getmark(int c, Buffer *buffer);
void clrmark(Line *line, Buffer *buffer);

// misccmds.cpp 
BOOL openfwd(BOOL split_line);
BOOL openbwd(void);
long cntllines(Line *pbegin,  Line *pend);
long plines(Xviwin *win, Line *lp);
long cntplines(Xviwin *win, Line *pbegin,  Line *pend);
Line *gotoline(Buffer *b,  unsigned long n);
int get_indent( Line *lp);
int set_indent(Line *lp,  int indent);
void tabinout(int inout, Line *start, Line *finish);
void makeargv(WCHAR *str, int *argcp, WCHAR ***argvp, WCHAR *whites);

// movement.cpp 
int shiftdown( Xviwin *win, unsigned nlines);
int shiftup( Xviwin *win, unsigned nlines);
void scrolldown( Xviwin *win, unsigned nlines);
void scrollup( Xviwin *win, unsigned nlines);
BOOL oneup( Xviwin *win, long nlines);
BOOL onedown( Xviwin *win, long nlines);
BOOL one_left(Xviwin *window, BOOL unused);
BOOL one_right(Xviwin *window, BOOL move_past_end);
void begin_line(Xviwin *window, BOOL flag);
void coladvance( Xviwin *win,  int col);
void do_goto(long line);
void move_cursor(Xviwin *win, Line *lp, int index);
void move_window_to_cursor( Xviwin *win);
void move_cursor_to_window(Xviwin *win);

// normal.cpp 
BOOL normal( int c);

// param.cpp 
void init_params(void);
void do_set(Xviwin *window, int argc, WCHAR *argv[], BOOL inter);
void parse_param(Xviwin *window, WCHAR *arg, BOOL inter);
void set_param(int n, ...);

// pipe.cpp 
void specify_pipe_range(Xviwin *window, Line *l1, Line *l2);
void do_pipe(Xviwin *window, WCHAR *command);

// preserve.cpp 
BOOL preservebuf(Xviwin *window);
BOOL do_preserve(void);

// ptrfunc.cpp 
enum mvtype inc( Posn *lp);
enum mvtype dec( Posn *lp);
void pswap( Posn *a,  Posn *b);
BOOL lt( Posn *a,  Posn *b);

// screen.cpp 
void init_screen(Xviwin *win);
void update_sline(Xviwin *win);
void update_cline(Xviwin *win);
void updateline(Xviwin *window);
void update_window(Xviwin *window);
void update_all(void);
void redraw_screen(void);
void clear(Xviwin *win);
void s_ins(Xviwin *win,  int row, int nlines);
void s_del( Xviwin *win, int row, int nlines);
void s_inschar(Xviwin *window, int newchar);
void wind_goto(Xviwin *win);
void cmd_init(Xviwin *win, int firstch);
Cmd_State cmd_input(Xviwin *win, int ch);
WCHAR *get_cmd(Xviwin *win);
void gotocmd(Xviwin *win, BOOL clr);
void prompt(WCHAR *message);
void beep( Xviwin *window);
void disp_init(Xviwin *win, WCHAR *(*func)(void), int colwidth, BOOL listmode);
BOOL disp_screen(Xviwin *win);

// search.cpp 
Posn *search(Xviwin *window, Line *startline, int startindex, int dir, WCHAR **strp);
Posn *nsearch(Xviwin *window, Line *startline, int startindex, int dir, WCHAR *str);
Line *linesearch(Xviwin *window, int dir, WCHAR **strp);
void regerror(WCHAR *s);
void do_global(Xviwin *window, Line *lp, Line *up, WCHAR *cmd, BOOL matchtype);
long do_substitute(Xviwin *window, Line *lp, Line *up, WCHAR *command);
long do_ampersand(Xviwin *window, Line *lp, Line *up, WCHAR *flags);
long do_tilde(Xviwin *window, Line *lp, Line *up, WCHAR *flags);

// startup.cpp 
Xviwin *xvi_startup(VirtScr *vs, int wargc, WCHAR *wargv[]);

// status.cpp 
void init_sline(Xviwin *win);
void show_message(Xviwin *window, WCHAR *format, ...);
void show_error(Xviwin *window, WCHAR *format, ...);
void show_file_info(Xviwin *window);

// tags.cpp 
void tagword(void);
BOOL do_tag(Xviwin *window, WCHAR *tag, BOOL force, BOOL interactive, BOOL split);

// undo.cpp 
void init_undo(Buffer *buffer);
BOOL start_command(Xviwin *window);
void end_command(Xviwin *window);
void replchars(Xviwin *window, Line *line, int start, int nchars, WCHAR *newstring);
void repllines( Xviwin *window, Line *line, long nolines, Line *newlines);
void replbuffer( Xviwin *window, Line *newlines);
void undo(Xviwin *window);
BOOL set_edit(Xviwin *window, Paramval new_value, BOOL interactive);

// version.cpp 
extern WCHAR Version[];

// windows.cpp 
Xviwin *init_window(VirtScr *vs);
Xviwin *split_window(Xviwin *oldwin);
void free_window(Xviwin *window);
void map_window_onto_buffer(Xviwin *w, Buffer *b);
void unmap_window(Xviwin *w);
Xviwin *next_window(Xviwin *window);
Xviwin *find_window(Xviwin *window, WCHAR *filename);
void resize_window(Xviwin *window, int nlines);
int move_sline(Xviwin *wp, int nlines);
void update_buffer(Buffer *buffer);
BOOL can_split(void);

// yankput.cpp 
void init_yankput(void);
BOOL do_yank(Buffer *buffer, Posn *from, Posn *to, BOOL charbased, int name);
BOOL yank_str(int name, WCHAR *str, BOOL line_based);
void do_put(Xviwin *win, Posn *location, int direction, int name);
void yp_stuff_input(Xviwin *win, int name, BOOL vi_mode);
void yp_push_deleted(void);
