// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)unicode.cpp		2.1 (Chris & John Downey) 7/29/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	unicode.cpp (ex. ascii.c)
* module function:
	Visual representations of control & other characters.

	This file is specific to the ASCII character set;
	versions for other character sets could be implemented if
	required.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

//
// Utility macro(s)
//

#define wsizeof(X) (sizeof(X)/sizeof(WCHAR))

//
// UNICODE --> Column width table.
//

static signed char chFontWidthTable[0x10000];

//
// Prepared stuffs for better performance of vischar().
//

static WCHAR ctrlstr[3];
static WCHAR tabstr[MAX_TABSTOP + 1];
static WCHAR nonprntstr[6];
static WCHAR w1[2];
static WCHAR w2[3];
static WCHAR w3[4];
static WCHAR w4[5];
static WCHAR hexdigit[17] = L"0123456789ABCDEF";
static HANDLE hConsoleDummy;

static void InitialzeDummyConsole()
{
	CONSOLE_SCREEN_BUFFER_INFO	csbi;
	COORD						coord;
	DWORD						dwWritten;

	hConsoleDummy = CreateConsoleScreenBuffer(GENERIC_WRITE | GENERIC_READ, 0, NULL
	, CONSOLE_TEXTMODE_BUFFER, NULL);
	if (hConsoleDummy == INVALID_HANDLE_VALUE) {
		API_FATAL(L"CreateConsoleScreenBuffer");
	}

	if ( ! GetConsoleScreenBufferInfo(hConsoleDummy, &csbi)) {
		API_FATAL(L"GetConsoleScreenBufferInfo");
	}

	// top left corner
	coord.X = 0;
	coord.Y = 0;

	// clear screen
	if ( ! FillConsoleOutputCharacter(hConsoleDummy, L' ', csbi.dwSize.Y * csbi.dwSize.X
	, coord, &dwWritten)) {
		API_FATAL(L"FillConsoleOutputCharacter");
	}
}

static int GetFontColumns(WCHAR wcUnicode)
{
	CONSOLE_SCREEN_BUFFER_INFO	csbi;
	COORD						coord;
	DWORD						dwWritten;

	// top left corner
	coord.X = 0;
	coord.Y = 0;

	if ( ! SetConsoleCursorPosition(hConsoleDummy, coord)) {
		API_FATAL(L"");
	}
	if ( ! WriteConsole(hConsoleDummy, &wcUnicode, 1, &dwWritten, NULL)) {
		API_FATAL(L"WriteConsole");
	}
	if ( ! GetConsoleScreenBufferInfo(hConsoleDummy, &csbi)) {
		API_FATAL(L"GetConsoleScreenBufferInfo");
	}
	return csbi.dwCursorPosition.X;
}

//
// init_vischar()
//
// Initialize function for vischar() in this module.
//
void init_vischar()
{
	int i;

	// Setup UNICODE Font Width Table(chFontWidthTable)

	// control characters
	for (i=0x0000; i<=0x001f; i++) {
		chFontWidthTable[i] = -1;
	}
	chFontWidthTable[0x0009] = -2;	// TAB
	chFontWidthTable[0x007f] = -3;	// DEL
	// basic ASCII characters
	for (i=0x0020; i<=0x007e; i++) {
		chFontWidthTable[i] = 1;
	}

	ctrlstr[0] = L'^';
	// crep[1] = ?
	ctrlstr[2] = L'\0';

	for (i = 0; i < MAX_TABSTOP; i++) tabstr[i] = L' ';

	nonprntstr[0] = L'\\';

	w2[1] =
	w3[1] = w3[2] =
	w4[1] = w4[2] = w4[3] = DUMMY_CHAR;

	InitialzeDummyConsole();
}

//
// vischar()
//
// Output visual representation for a character. Return value is the
// width, in display columns, of the representation; the actual string
// is returned in *pp unless pp is NULL.
//
// If tabcol is -1, a tab is represented as L"^I"; otherwise, it gives
// the current physical column, & is used to determine the number of
// spaces in the string returned as *pp.
//
// Looks at the parameters tabs, list and cchars.
// This may sound costly, but in fact it's a single memory
// access for each parameter.
//
// This function is *BOTTLE-NECK* of scrolling speed.
// DON'T WRITE SLOW CODE !

unsigned vischar(int c, WCHAR **pp, int tabcol)
{
retry:
	switch (chFontWidthTable[c]) {

non_printable_code:
	case -4:		// non-printable code ... "\0000" - "\FFFF"
		if (pp != NULL) {
			*pp = nonprntstr;
			nonprntstr[1] = hexdigit[((c >> 12) & 0x000f)];
			nonprntstr[2] = hexdigit[((c >>  8) & 0x000f)];
			nonprntstr[3] = hexdigit[((c >>  4) & 0x000f)];
			nonprntstr[4] = hexdigit[( c        & 0x000f)];
		}
		return 5;

	case -3:	// DEL ... "^?"

		if (Pb(P_cchars)) goto printable_code1;
		if (pp != NULL) {
			*pp = ctrlstr;
			ctrlstr[1] = L'?';
		}
		return 2;

	case -2:	// TAB
		if (tabcol >= 0 && Pb(P_tabs) && !Pb(P_list)) {
			// Tab which we have to display as a string of
			// spaces (rather than "^I" or directly).
			unsigned	nspaces;

			while (tabcol >= Pn(P_tabstop))
				tabcol -= Pn(P_tabstop);
			nspaces = Pn(P_tabstop) - tabcol;
			if (pp != NULL) {
				*pp = &tabstr[MAX_TABSTOP - nspaces];
			}
			return nspaces;
		}
		// no break;

	case -1:	// ASCII control code ... "^A" etc.
		if (Pb(P_cchars)) goto printable_code1;
		if (pp != NULL) {
			*pp = ctrlstr;
			ctrlstr[1] = c + L'@';
		}
		return 2;

	case 0:		// Font Width is unknown ...
		if ( (chFontWidthTable[c] = GetFontColumns(c)) <= 0) {
			chFontWidthTable[c] = -4;
		}
		goto retry;

printable_code1:
	case 1:		// one column (for most latin code)
		if (pp != NULL) {
			*pp = w1;
			w1[0] = c;
		}
		return 1;

	case 2:		// two columns (for CJK etc.)
		if (pp != NULL) {
			*pp = w2;
			w2[0] = c;
		}
		return 2;

	case 3:		// three columns (OOPS !)
		if (pp != NULL) {
			*pp = w3;
			w3[0] = c;
		}
		return 3;

	case 4:		// four columns (Oh my, my...)
		if (pp != NULL) {
			*pp = w4;
			w4[0] = c;
		}
		return 4;

	default:
		goto non_printable_code;	// give up !
	}
}
