// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)param.c		2.1 (Chris & John Downey) 7/29/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	param.c
* module function:
	Code to handle user-settable parameters. This is all pretty much
	table-driven. To add a new parameter, put it in the params array,
	and add a macro for it in param.h.

	The idea of the parameter table is that access to any particular
	parameter has to be fast, so it is done with a table lookup. This
	unfortunately means that the index of each parameter is recorded
	as a macro in param.h, so that file must be changed at the same
	time as the table below, and in the same way.

	When a parameter is changed, a function is called to do the actual
	work; this function is part of the parameter structure.  For many
	parameters, it's just a simple function that prints "not implemented";
	for most others, there are "standard" functions to set bool, numeric
	and string parameters, with a certain amount of checking.

	No bounds checking is done here; we should really include limits
	to numeric parameters in the table. Maybe this will come later.

	The data structures will be changed again shortly to enable
	buffer- and window-local parameters to be implemented.

	One problem with numeric parameters is that they are of type "int";
	this obviously places some restrictions on the sort of things they
	may be used for, and it may be necessary at some point to change
	this type to something like "unsigned long".

* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

#define nofunc	NULL

// Default settings for string parameters.
// These are set by the exported function init_params(),
// which must be called before any parameters are accessed.
#define DEF_TAGS		L"tags,c:\\usr\\lib\\tags"
#define DEF_PARA		L"^($|\\.([ILPQ]P|LI|[plib]p))"
#define DEF_SECTIONS	L"^({|\\.([NS]H|HU|nh|sh))"
#define DEF_SENTENCES	L"\\<[A-Z]"

// Default settings for showing control- and meta-characters are
// as for "normal" vi, i.e. "old" xvi without SHOW_META_CHARS set.
#ifndef DEF_CCHARS
#	define		DEF_CCHARS		FALSE
#endif
// Meta-characters is nonsense for UNICODE version.
//#ifndef DEF_MCHARS
//#	define		DEF_MCHARS		FALSE
//#endif

// Internal functions.
static	int 	strtoi(WCHAR **);
static	BOOL	_do_set(Xviwin *, WCHAR *, BOOL);
static	WCHAR	*parmstring(Param *, int);
static	void	enum_usage(Xviwin*, Param *);
static	BOOL	not_imp(Xviwin *, Paramval, BOOL);
static	BOOL	set_magic(Xviwin *, Paramval, BOOL);
static	BOOL	set_rt(Xviwin *, Paramval, BOOL);
static	WCHAR	*par_show(void);

// These are the available parameters. The following are non-standard:
//
//		autosplit format helpfile jumpscroll preserve
//		preservetime regextype vbell edit
//		color statuscolor roscolor systemcolor
Param	params[] = {
//		fullname		shortname		flags		value			function ... 
	{	L"ada",			L"ada",			P_BOOL, 	0,				not_imp,   },
	{	L"adapath",		L"adapath",		P_STRING,	0,				not_imp,   },
	{	L"autoindent",	L"ai",			P_BOOL, 	0,				nofunc,    },
	{	L"autoprint",	L"ap",			P_BOOL, 	0,				not_imp,   },
	{	L"autosplit",	L"as",			P_NUM,		2,				nofunc,    },
	{	L"autowrite",	L"aw",			P_BOOL, 	0,				not_imp,   },
	{	L"beautify", 	L"bf",			P_BOOL, 	0,				not_imp,   },
	{	L"cchars",		L"cchars",		P_BOOL, 	DEF_CCHARS, 	nofunc,    },
	{	L"codepage",	L"cp",			P_NUM,		0,				set_codepage,},
	{	L"codetype",	L"ct",			P_ENUM,		0, 				set_codetype,},
	{	L"color",		L"co",			P_NUM,		0, 				nofunc,    },
	{	L"directory",	L"dir",			P_STRING,	0,				not_imp,   },
	{	L"edcompatible", L"edcompatible", P_BOOL, 	0,				not_imp,   },
	{	L"edit", 		L"edit", 		P_BOOL, 	TRUE,			set_edit,  },
	{	L"errorbells",	L"eb",			P_BOOL, 	0,				nofunc,    },
	{	L"extendedkey",	L"ek",			P_BOOL, 	TRUE,			nofunc,    },
	{	L"format",		L"fmt",			P_ENUM, 	0,				set_format,},
	{	L"hardtabs", 	L"ht",			P_NUM,		0,				not_imp,   },
	{	L"helpfile", 	L"hf",			P_STRING,	0,				nofunc,    },
	{	L"ignorecase",	L"ic",			P_BOOL, 	0,				nofunc,    },
	{	L"jumpscroll",	L"js",			P_ENUM, 	0,				nofunc,    },
	{	L"lisp", 		L"lisp", 		P_BOOL, 	0,				not_imp,   },
	{	L"list", 		L"ls",			P_BOOL, 	0,				nofunc,    },
	{	L"magic",		L"magic",		P_BOOL, 	TRUE,			set_magic, },
//	{	L"mchars",		L"mchars",		P_BOOL, 	DEF_MCHARS, 	nofunc,    },
	{	L"mesg", 		L"mesg", 		P_BOOL, 	0,				not_imp,   },
	{	L"minrows",		L"min",			P_NUM,		2,				nofunc,    },
	{	L"modeline", 	L"modeline", 	P_BOOL, 	0,				not_imp,   },
	{	L"number",		L"nu",			P_BOOL, 	0,				nofunc,    },
	{	L"open", 		L"open", 		P_BOOL, 	0,				not_imp,   },
	{	L"optimize", 	L"opt",			P_BOOL, 	0,				not_imp,   },
	{	L"paragraphs",	L"para", 		P_STRING,	0,				nofunc,    },
	{	L"preserve", 	L"psv",			P_ENUM, 	0,				nofunc,    },
	{	L"preservetime", L"psvt", 		P_NUM,		5,				nofunc,    },
	{	L"prompt",		L"prompt",		P_BOOL, 	0,				not_imp,   },
	{	L"readonly", 	L"ro",			P_BOOL, 	0,				nofunc,    },
	{	L"redraw",		L"redraw",		P_BOOL, 	0,				not_imp,   },
	{	L"regextype",	L"rt",			P_ENUM, 	0,				set_rt,    },
	{	L"remap",		L"remap",		P_BOOL, 	0,				nofunc,    },
	{	L"report",		L"report",		P_NUM,		5,				nofunc,    },
	{	L"roscolor",	L"rst",			P_NUM,		0,				nofunc,    },
	{	L"scroll",		L"scroll",		P_NUM,		0,				not_imp,   },
	{	L"sections", 	L"sections", 	P_STRING,	0,				nofunc,    },
	{	L"sentences",	L"sentences",	P_STRING,	0,				nofunc,    },
	{	L"shell",		L"sh",			P_STRING,	0,				nofunc,    },
	{	L"shiftwidth",	L"sw",			P_NUM,		8,				nofunc,    },
	{	L"showmatch",	L"sm",			P_BOOL, 	0,				nofunc,    },
	{	L"slowopen", 	L"slowopen", 	P_BOOL, 	0,				not_imp,   },
	{	L"sourceany",	L"sourceany",	P_BOOL, 	0,				not_imp,   },
	{	L"statuscolor", L"st",			P_NUM,		0,				nofunc,    },
	{	L"systemcolor", L"sy",			P_NUM,		0,				nofunc,    },
	{	L"tabs", 		L"tabs", 		P_BOOL, 	TRUE,			nofunc,    },
	{	L"tabstop",		L"ts",			P_NUM,		8,				nofunc,    },
	{	L"taglength",	L"tlh",			P_NUM,		0,				nofunc,    },
	{	L"tags", 		L"tags", 		P_LIST, 	0,				nofunc,    },
	{	L"term", 		L"term", 		P_STRING,	0,				not_imp,   },
	{	L"terse",		L"terse",		P_BOOL, 	0,				not_imp,   },
	{	L"timeout",		L"timeout",		P_NUM,		DEF_TIMEOUT,	nofunc,    },
	{	L"ttytype",		L"ttytype",		P_STRING,	0,				not_imp,   },
	{	L"vbell",		L"vb",			P_BOOL, 	0,				nofunc,    },
	{	L"warn", 		L"warn", 		P_BOOL, 	0,				not_imp,   },
	{	L"window",		L"window",		P_NUM,		0,				not_imp,   },
	{	L"wrapmargin",	L"wm",			P_NUM,		0,				nofunc,    },
	{	L"wrapscan", 	L"ws",			P_BOOL, 	TRUE,			nofunc,    },
	{	L"writeany", 	L"wa",			P_BOOL, 	0,				not_imp,   },

	{	(WCHAR *) NULL,	(WCHAR *) NULL,	0,			0,				nofunc,    },

};

// Special initialisations for string, list and enum parameters,
// which we cannot put in the table above because C does not
// allow the initialisation of unions.
static struct {
	int 		index;
	WCHAR		*value;
} init_str[] = {		// strings and lists 
	P_helpfile, 		HELPFILE,
	P_paragraphs,		DEF_PARA,
	P_sections, 		DEF_SECTIONS,
	P_sentences,		DEF_SENTENCES,
	P_tags, 			DEF_TAGS,
};
#define NSTRS	(sizeof(init_str) / sizeof(init_str[0]))

// Names of values for the P_jumpscroll enumerated parameter.
//
// It is essential that these are in the same order as the js_...
// symbolic constants defined in xvi.h.
static WCHAR *js_strings[] =
{
	L"off",				// js_OFF 
	L"auto", 			// js_AUTO 
	L"on",				// js_ON 
	NULL
};

static struct {
	int 		index;
	int 		value;
	WCHAR		**elist;
} init_enum[] = {		// enumerations 
	P_codetype,			DEF_TCT,		codetype_strings,
	P_format,			DEF_TFF,		fmt_strings,
	P_jumpscroll,		js_AUTO,		js_strings,
	P_preserve, 		psv_STANDARD,	psv_strings,
	P_regextype,		rt_GREP,		rt_strings,
};
#define NENUMS	(sizeof(init_enum) / sizeof(init_enum[0]))

// These are used by par_show().
static	BOOL	show_all;
static	Param	*curparam;

// Initialise parameters.
//
// This function is called once from startup().
void init_params(void)
{
	Paramval	pv;
	Param		*pp;
	int 		i;

	// First go through the special string and enum initialisation
	// tables, setting the values into the union field in the
	// parameter structures.
	for (i = 0; i < NSTRS; i++) {
		set_param(init_str[i].index, init_str[i].value);
	}
	for (i = 0; i < NENUMS; i++) {
		set_param(init_enum[i].index, init_enum[i].value,
						init_enum[i].elist);
	}

	// Get default codepage.
	set_param(P_codepage, get_system_codepage());

	// Set default color information.
	set_default_colors();

	// Call any special functions that have been set up for
	// any parameters, using the default values included in
	// the parameter table.
	for (pp = &params[0]; pp->p_fullname != NULL; pp++) {
		if (pp->p_func != nofunc && pp->p_func != not_imp) {
			switch (pp->p_flags & P_TYPE) {
			case P_NUM:
			case P_ENUM:
				pv.pv_i = pp->p_value;
				break;
			case P_BOOL:
				pv.pv_b = pp->p_value;
				break;
			case P_STRING:
				pv.pv_s = pp->p_str;
				break;
			case P_LIST:
				pv.pv_l = pp->p_list;
			}
			(void) (*pp->p_func)(curwin, pv, FALSE);
		}
	}
}

void do_set(Xviwin *window, int argc, WCHAR *argv[], BOOL inter)
{
	int count;

	// First check to see if there were any parameters to set,
	// or if the user just wants us to display the parameters.
	if (argc == 0 || (argc == 1 && (argv[0][0] == L'\0' ||
								wcsncmp(argv[0], L"all", 3) == 0))) {
		if (inter) {
			int pcwidth;

			show_all = (argc != 0 && argv[0][0] != L'\0');
			curparam = &params[0];
			pcwidth = (window->w_ncols < 90 ?
						window->w_ncols :
						(window->w_ncols < 135 ?
							window->w_ncols / 2 :
							window->w_ncols / 3));

			disp_init(window, par_show, pcwidth, FALSE);
		}

		return;
	}

	for (count = 0; count < argc; count++) {
		if (!_do_set(window, argv[count], inter)) {
			break;
		}
	}

	if (inter) {

		// Finally, update the screen in case we changed
		// something like "tabstop" or "list" that will change
		// its appearance. We don't always have to do this,
		// but it's easier for now.
		update_all();
	}
}

void parse_param(Xviwin *window, WCHAR *arg, BOOL inter)
{
	WCHAR *p;

	p = wcstok(arg, L" \t");
	if (p != NULL && wcscmp(p, L"set") == 0) p = wcstok(NULL, L" \t");
	for ( ; p != NULL; p = wcstok(NULL, L" \t")) {
		_do_set(window, p, inter);
	}
	if (inter) {
		update_all();
	}
}

// Convert a string to an integer. The string may encode the integer
// in octal, decimal or hexadecimal form, in the same style as C. On
// return, make the string pointer, which is passed to us by
// reference, point to the first character which isn't valid for the
// base the number seems to be in.
static int strtoi(WCHAR **sp)
{
	WCHAR		*s;
	int		i, c;
	BOOL		neg;

	i = 0;
	neg = FALSE;
	s = *sp;
	c = *s;
	while (iswspace(c)) {
		c = *++s;
	}
	if (c == L'-') {
		neg = TRUE;
		c = *++s;
	}
	while (iswspace(c)) {
		c = *++s;
	}
	if (c == L'0') {
		switch (c = *++s) {
		case L'x': case L'X':
			// We've got 0x ... or 0X ..., so it
			// looks like a hex. number.
			while ((c = *++s) != L'\0' && iswxdigit(c)) {
				i = (i * 16) + hex_to_bin(c);
			}
			break;

		case L'0': case L'1': case L'2': case L'3':
		case L'4': case L'5': case L'6': case L'7':
			// It looks like an octal number.
			do {
				i = (i * 8) + c - L'0';
			} while ((c = *++s) != L'\0' && iswoctdigit(c));
			break;

		default:
			*sp = s;
			return(0);
		}
	} else {
		// Assume it's decimal.
		while (c != L'\0' && iswdigit(c)) {
			i = (i * 10) + c - L'0';
			c = *++s;
		}
	}
	*sp = s;
	return(neg ? -i : i);
}

// Internal version of do_set(). Returns TRUE if set of specified
// parameter was successful, otherwise FALSE.
static BOOL _do_set(Xviwin *window, WCHAR *arg, BOOL inter)
{
	int 		settype = 0;			// type they want to set 
	BOOL		bool_value; 			// value they want to set 
	WCHAR		*str_value; 			// value they want to set 
	Param		*pp;
	Paramval	val;
	WCHAR		*cp;
	WCHAR		**ep;


	// "arg" points at the parameter we are going to try to set.
	// Spot "no" and "=" keys.

	// Note that we will have to change this later if we
	// want to implement "nomatch" and "nonomatch". :-(
	if (wcsncmp(arg, L"no", 2) == 0) {
		settype = P_BOOL;
		bool_value = FALSE;
		arg += 2;
	} else {
		bool_value = TRUE;
	}

	str_value = wcschr(arg, L'=');
	if (str_value == NULL) str_value = wcschr(arg, L':');
	if (str_value != NULL) {
		if (settype == P_BOOL) {
			if (inter) {
				show_error(window, L"Can't give \"no\" and a value");
				return(FALSE);
			}
		}

		// Null-terminate the parameter name
		// and point str_value at the value to
		// the right of the '='.
		*str_value++ = L'\0';
		settype = P_STRING;

	} else if (settype != P_BOOL) {

		// Not already set to P_BOOL, so we haven't seen a "no".
		// No '=' sign, so assume we are setting a boolean
		// parameter to TRUE.
		settype = P_BOOL;
		bool_value = TRUE;
	}

	// Now search for a complete match of the parameter
	// name with either the full or short name in the
	// parameter table.
	for (pp = &params[0]; pp->p_fullname != NULL; pp++) {
		if (wcscmp(arg, pp->p_fullname) == 0 ||
			wcscmp(arg, pp->p_shortname) == 0)
			break;
	}

	if (pp->p_fullname == NULL) {				// no match found 
		if (inter) {
			show_error(window, L"No such parameter");
			return(FALSE);
		}
	}

	// Check the passed type is appropriate for the
	// parameter's type. If it isn't, winge and return.
	switch (pp->p_flags & P_TYPE) {
	case P_STRING:
	case P_LIST:
	case P_NUM:
		if (settype != P_STRING) {
			if (inter) {
				show_error(window,
					(pp->p_flags & P_STRING) ?
					L"Invalid set of string parameter"
					: (pp->p_flags & P_NUM) ?
					L"Invalid set of numeric parameter"
					: // else 
					L"Invalid set of list parameter");
			}
			return(FALSE);
		}
		break;

	case P_ENUM:
		if (settype != P_STRING) {
			if (inter) {
				enum_usage(window, pp);
			}
			return(FALSE);
		}
		break;

	case P_BOOL:
		if (settype != P_BOOL) {
			if (inter) {
				show_error(window, L"Invalid set of boolean parameter");
			}
			return(FALSE);
		}
	}

	// Do any type-specific checking, and set up the
	// "val" union to contain the (decoded) value.
	switch (pp->p_flags & P_TYPE) {
	case P_NUM:
	{
		int i;

		cp = str_value;
		i = strtoi(&cp);
		// If there are extra characters after the number,
		// don't accept it.
		if (*cp != L'\0') {
			if (inter) {
				show_error(window, L"Invalid numeric parameter");
			}
			return(FALSE);
		}

		val.pv_i = i;
		break;
	}

	case P_ENUM:
		for (ep = pp->p_elist; *ep != NULL; ep++) {
			if (wcscmp(*ep, str_value) == 0)
				break;
		}

		if (*ep == NULL) {
			if (inter) {
				enum_usage(window, pp);
			}
			return(FALSE);
		}

		val.pv_i = ep - pp->p_elist;
		break;

	case P_STRING:
	case P_LIST:
		val.pv_s = str_value;
		break;

	case P_BOOL:
		val.pv_b = bool_value;
		break;
	}

	// Call the check function if there is one.
	if (pp->p_func != nofunc && (*pp->p_func)(window, val, inter) == FALSE) {
		return(FALSE);
	}

	// Set the value.
	switch (pp->p_flags & P_TYPE) {
	case P_NUM:
	case P_ENUM:
		pp->p_value = val.pv_i;
		break;

	case P_BOOL:
		pp->p_value = bool_value;
		break;

	case P_STRING:
	case P_LIST:
		set_param(pp - params, str_value);
		break;
	}
	pp->p_flags |= P_CHANGED;

	// Check the bounds for numeric parameters here.
	// We will have to make this table-driven later.
	if (Pn(P_tabstop) <= 0 || Pn(P_tabstop) > MAX_TABSTOP) {
		if (inter) {
			show_error(window, L"Invalid tab size specified");
		}
		set_param(P_tabstop, 8);
		return(FALSE);
	}

	// Got through all the bounds checking, so we're okay.
	return TRUE;
}

// Internal parameter setting - parameters are not considered
// to have been changed unless they are set by the user.
//
// All types of parameter are handled by this function.
// Enumerated types are special, because two arguments
// are expected: the first, always present, is the index,
// but the second may be either a valid list of strings
// a NULL pointer, the latter indicating that the list
// of strings is not to be changed.
//
// The P_LIST type accepts a string argument and converts
// it to a vector of strings which is then stored.
void set_param(int n, ...)
{
	va_list 			argp;
	Param				*pp;
	WCHAR				*cp;
	WCHAR				**elist;

	pp = &params[n];

	va_start(argp, n);

	switch (pp->p_flags & P_TYPE) {
	case P_ENUM:
		pp->p_value = va_arg(argp, int);

		// If the second argument is a non-NULL list of
		// strings, set it into the parameter structure.
		// Note that this is not dependent on the return
		// value from the check function being TRUE,
		// since the check cannot be made until the
		// array of strings is in place.
		elist = va_arg(argp, WCHAR **);
		if (elist != NULL) {
			pp->p_elist = elist;
		}
		break;

	case P_BOOL:
	case P_NUM:
		pp->p_value = va_arg(argp, int);
		break;

	case P_LIST:
		{
			int argc;
			WCHAR		**argv;

			cp = strsave(va_arg(argp, WCHAR *));
			if (cp == NULL) {
				// This is not necessarily a good idea.
				show_error(curwin, L"Can't allocate memory for parameter");
				return;
			}

			makeargv(cp, &argc, &argv, L" \t,");
			if (argc == 0 || argv == NULL) {
				free(cp);
				return;
			}
			if (pp->p_list != NULL) {
				if (pp->p_list[0] != NULL) {
					free(pp->p_list[0]);
				}
				free((WCHAR *) pp->p_list);
			}
			pp->p_list = argv;
		}
		break;

	case P_STRING:
		cp = strsave(va_arg(argp, WCHAR *));
		if (cp == NULL) {
			// This is not necessarily a good idea.
			show_error(curwin, L"Can't allocate memory for parameter");
		} else {
			// We always free up the old string, because
			// it must have been allocated at some point.
			if (pp->p_str != NULL) {
				free(pp->p_str);
			}
			pp->p_str = cp;
		}
		break;
	}

	va_end(argp);
}

// Display helpful usage message for an enumerated parameter, listing
// the legal values for it.
static void enum_usage(Xviwin *w, Param *pp)
{
	Flexbuf 			s;
	WCHAR				**sp;

	flexnew(&s);
	(void) lformat(&s, L"Must be one of:");
	for (sp = (WCHAR **) pp->p_str; *sp != NULL; sp++) {
		(void) lformat(&s, L" %s", *sp);
	}
	show_error(w, L"%s", flexgetstr(&s));
	flexdelete(&s);
}

// Return a string representation for a single parameter.
static WCHAR *parmstring(Param *pp, int leading)
{
	static Flexbuf		b;

	flexclear(&b);
	while (leading-- > 0) {
		(void) flexaddch(&b, L' ');
	}
	switch (pp->p_flags & P_TYPE) {
	case P_BOOL:
		(void) lformat(&b, L"%s%s", (pp->p_value ? L"" : L"no"), pp->p_fullname);
		break;

	case P_NUM:
		(void) lformat(&b, L"%s=%d", pp->p_fullname, pp->p_value);
		break;

	case P_ENUM:
	{
		int 	n;
		WCHAR	*estr;

		for (n = 0; ; n++) {
			if ((estr = ((WCHAR **) pp->p_str)[n]) == NULL) {
				estr = L"INTERNAL ERROR";
				break;
			}
			if (n == pp->p_value)
				break;
		}
		(void) lformat(&b, L"%s=%s", pp->p_fullname, estr);
		break;
	}

	case P_STRING:
		(void) lformat(&b, L"%s=%s", pp->p_fullname,
									(pp->p_str != NULL) ? pp->p_str : L"");
		break;

	case P_LIST:
		{
			WCHAR		**cpp;

			(void) lformat(&b, L"%s=%s", pp->p_fullname, pp->p_list[0]);
			for (cpp = pp->p_list + 1; *cpp != NULL; cpp++) {
				(void) lformat(&b, L" %s", *cpp);
			}
		}
		break;
	}
	return(flexgetstr(&b));
}

static WCHAR *par_show(void)
{
	static BOOL		started = FALSE;

	if (!started) {
		started = TRUE;
		return(L"Parameters:");
	}

	for ( ; curparam->p_fullname != NULL; curparam++) {

		// Ignore unimplemented parameters.
		if (curparam->p_func != not_imp) {
			// Display all parameters if show_all is set;
			// otherwise, just display changed parameters.
			if (show_all || (curparam->p_flags & P_CHANGED) != 0) {
				break;
			}
		}
	}

	if (curparam->p_fullname == NULL) {
		started = FALSE;				// reset for next time 
		return(NULL);
	} else {
		WCHAR	*retval;

		retval = parmstring(curparam, 3);
		curparam++;
		return(retval);
	}
}

static BOOL not_imp(Xviwin *window, Paramval new_value, BOOL interactive)
{
	if (interactive) {
		show_message(window, L"That parameter is not implemented!");
	}
	return(TRUE);
}

static BOOL set_magic(Xviwin *window, Paramval new_value, BOOL interactive)
{
	static int	prev_rt = rt_GREP;

	if (new_value.pv_b) {
		// Turn magic on.
		if (Pn(P_regextype) == rt_TAGS) {
			set_param(P_regextype, prev_rt, (WCHAR **) NULL);
		}
	} else {
		// Turn magic off.
		if (Pn(P_regextype) != rt_TAGS) {
			prev_rt = Pn(P_regextype);
			set_param(P_regextype, rt_TAGS, (WCHAR **) NULL);
		}
	}
	return(TRUE);
}

static BOOL set_rt(Xviwin *window, Paramval new_value, BOOL interactive)
{
	switch (new_value.pv_i) {
	case rt_TAGS:
	case rt_GREP:
	case rt_EGREP:
		set_param(P_magic, (new_value.pv_i != rt_TAGS));
		return(TRUE);
	default:
		return(FALSE);
	}
}
