// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)misccmds.c	2.2 (Chris & John Downey) 8/28/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	misccmds.c
* module function:
	Miscellaneous functions.

	This module will probably get hacked later and split
	up more sensibly.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

// Add a blank line above or below the current line.
// Returns TRUE for success, FALSE for failure to get memory.
//
// The single boolean parameter tells us whether to split the
// current line at the cursor position, or just to open a new
// line leaving the current one intact.
BOOL openfwd(BOOL split_line)
{
	Line				*l; 			// pointer to newly allocated line 
	Posn		*oldposn;
	Line		*oldline;
	WCHAR		*otext;

	oldposn = curwin->w_cursor;
	oldline = oldposn->p_line;
	otext = oldline->l_text;

	// First find space for new line.
	//
	// By asking for as much space as the prior line had we make sure
	// that we'll have enough space for any auto-indenting.
	l = newline(wcslen(otext) + SLOP);
	if (l == NULL)
		return(FALSE);

	// Link the new line into the list.
	repllines(curwin, oldline->l_next, 0L, l);

	// Do auto-indent.
	if (Pb(P_autoindent)) {
		*l->l_text = L'\0';
		indentchars = set_indent(l, get_indent(oldline));
	} else {
		indentchars = 0;
	}

	// If we're in insert mode, we need to move the remainder of the
	// current line onto the new line. Otherwise the new line is left
	// blank.
	if (split_line) {
		WCHAR	*s;

		s = otext + oldposn->p_index;

		replchars(curwin, l, indentchars, 0, s);
		replchars(curwin, oldline, oldposn->p_index, wcslen(s), L"");
	}

	// Move cursor to the new line.
	move_cursor(curwin, l, indentchars);
	move_window_to_cursor(curwin);
	cursupdate(curwin);
	update_buffer(curbuf);

	return(TRUE);
}

// Add a blank line above the current line.
// Returns TRUE for success, FALSE for failure to get memory.
BOOL openbwd(void)
{
	Line				*l;
	Line		*oldline;
	WCHAR		*otext;

	oldline = curwin->w_cursor->p_line;
	otext = oldline->l_text;

	// First find space for new line.
	l = newline(wcslen(otext) + SLOP);
	if (l == NULL)
		return(FALSE);

	// Link the new line into the list.
	repllines(curwin, oldline, 0L, l);

	// Do auto-indent.
	if (Pb(P_autoindent)) {
		*l->l_text = L'\0';
		indentchars = set_indent(l, get_indent(oldline));
	} else {
		indentchars = 0;
	}

	// Ensure the cursor is pointing at the right line.
	move_cursor(curwin, l, indentchars);
	move_window_to_cursor(curwin);
	cursupdate(curwin);
	update_buffer(curbuf);

	return(TRUE);
}

// Count the number of lines between the two given lines.
// If the two given lines are the same, the return value
// is 1, not 0; i.e. the count is inclusive.
//
// Note that this function has been changed to give the
// correct number of lines, even if they are ordered wrongly.
// This change is backwards-compatible with the old version.
long cntllines(Line *pbegin,  Line *pend)
{
	Line		*lp;
	long		lnum;
	BOOL				swapped = FALSE;

	// Ensure correct ordering.
	if (later(pbegin, pend)) {
		lp = pbegin;
		pbegin = pend;
		pend = lp;
		swapped = TRUE;
	}

	for (lnum = 1, lp = pbegin; lp != pend; lp = lp->l_next) {
		lnum++;
	}

	if (swapped)
		lnum = - lnum;

	return(lnum);
}

// plines(lp) - return the number of physical screen lines taken by line 'lp'.
long plines(Xviwin *win, Line *lp)
{
	long		col;
	WCHAR		*s;

	s = lp->l_text;

	if (*s == L'\0') 			// empty line 
		return(1);

	// If list mode is on, then the '$' at the end of
	// the line takes up one extra column.
	col = Pb(P_list) ? 1 : 0;

	if (Pb(P_number)) {
		col += NUM_SIZE;
	}

	for ( ; *s != L'\0'; s++) {
		col += vischar(*s, (WCHAR **) NULL, (int) col);
	}

	{
		int	row;
		int	columns;

		columns = win->w_ncols;
		for (row = 1; col > columns; ) {
			row++;
			col -= columns;
		}
		return row;
	}
}

// Count the number of physical lines between the two given lines.
//
// This routine is like cntllines(), except that:
//		it counts physical rather than logical lines
//		it always returns the absolute number of physical lines
//		it is non-inclusive
//		if the physical line count for a group of lines is greater
//		than or equal to rows * 2, we just return rows * 2; we assume
//		the caller isn't interested in the exact number.
long cntplines(Xviwin *win, Line *pbegin,  Line *pend)
{
	Line		*lp;
	long		physlines;
	unsigned			toomuch;

	// Ensure correct ordering.
	if (later(pbegin, pend)) {
		lp = pbegin;
		pbegin = pend;
		pend = lp;
	}

	toomuch = win->w_nrows * 2;
	for (physlines = 0, lp = pbegin; lp != pend; lp = lp->l_next) {
		physlines += plines(win, lp);
		if (physlines >= toomuch)
			break;
	}

	return(physlines);
}

// gotoline(buffer, n) - return a pointer to line 'n' in the given buffer
//
// Returns the first line of the file if n is 0.
// Returns the last line of the file if n is beyond the end of the file.
Line *gotoline(Buffer *b,  unsigned long n)
{
	if (n == 0) {
		return(b->b_file);
	} else {
		Line	*lp;

		for (lp = b->b_file; --n > 0 && lp->l_next != b->b_lastline;
														lp = lp->l_next) {
			;
		}
		return(lp);
	}
}

int get_indent( Line *lp)
{
	WCHAR	*text;
	int	indent;
	int	ts = Pn(P_tabstop); // synonym for efficiency 

	if (lp == NULL || (text = lp->l_text) == NULL) {
		show_error(curwin, L"Internal error: get_indent(NULL)");
		return 0;
	}

	for (indent = 0; *text != L'\0' && (*text == L' ' || *text == L'\t');
																text++) {
		indent += *text == L' ' ? 1 : ts - indent % ts;
	}
	return indent;
}

// Set number of columns of leading whitespace on line, regardless of
// what was there before, & return number of characters (not columns)
// used.
int set_indent(Line *lp,  int indent)
{
	WCHAR		*cp;			// temp char pointer for loops 
	int			ntabs;			// number of tabs to use 
	unsigned	nnew;			// no of chars used in old line 
	unsigned	nold;			// no of chars used in new version 
	WCHAR		*newstr;		// allocated string for new indent 

	if (lp == NULL || lp->l_text == NULL) {
		show_error(curwin, L"Internal error: set_indent(0)");
		return(0);
	}

	// Find out how many tabs we need, & how many spaces.
	for (ntabs = 0; indent >= Pn(P_tabstop); ntabs++)
		indent -= Pn(P_tabstop);

	// Find out how many characters were used for initial
	// whitespace in the current (old) line.
	for (cp = lp->l_text; *cp == L' ' || *cp == L'\t'; cp++) {
		;
	}
	nold = cp - lp->l_text;

	// "nnew" is the number of characters we will use
	// for indentation in the new version of the line.
	nnew = ntabs + indent;

	// Get some space, and place into it the string of tabs
	// and spaces which will form the new indentation.
	// If no space available, return nold as we have not
	// changed the line; this is the correct action.
	newstr = alloc_wchar((unsigned) nnew + 1);
	if (newstr == NULL)
		return(nold);

	cp = newstr;
	while (ntabs-- > 0)
		*cp++ = L'\t';
	while (indent-- > 0)
		*cp++ = L' ';
	*cp = L'\0';

	// Finally, replace the old with the new.
	replchars(curwin, lp, 0, (int) nold, newstr);

	free(newstr);

	return(nnew);
}

// tabinout(inout, start, finish)
//
// "inout" is either '<' or '>' to indicate left or right shift.
void tabinout(int inout, Line *start, Line *finish)
{
	Line		*lp;
	long		nlines = 0;

	if (!start_command(curwin)) {
		return;
	}

	finish = finish->l_next;

	for (lp = start; lp != finish; lp = lp->l_next) {
		WCHAR *p;

		// Find out whether it's a blank line (either
		// empty or containing only spaces & tabs).
		// If so, just remove all whitespace from it.
		for (p = lp->l_text; *p && (*p == L'\t' || *p == L' '); p++)
			;
		if (*p == L'\0') {
			if (p > lp->l_text) {
				replchars(curwin, lp, 0, p - lp->l_text, L"");
			}
		} else if (inout == L'<') {
			int oldindent = get_indent(lp);

			(void) set_indent(lp, (oldindent <= Pn(P_shiftwidth) ?
									0 : oldindent - Pn(P_shiftwidth)));
		} else {
			(void) set_indent(lp, get_indent(lp) + Pn(P_shiftwidth));
		}

		nlines++;
	}

	end_command(curwin);

	if (nlines > Pn(P_report)) {
		show_message(curwin, L"%ld lines %ced", nlines, inout);
	}
}

// Construct a vector of pointers into each word in
// the given string. Intervening whitespace will be
// converted to null bytes.
//
// Returned vector is constructed in allocated space,
// and so must be freed after use.
//
// An extra NULL pointer is always allocated for safety.
//
// If memory cannot be allocated, or if there are no
// words in the given string, or if it is a NULL ptr,
// then the returned values will be 0 and NULL.
//
// The "whites" argument is a pointer to an array of
// characters which are to be considered as whitespace.
void makeargv(WCHAR *str, int *argcp, WCHAR ***argvp, WCHAR *whites)
{
	int 		argc;
	WCHAR		**argv;
	int 		argv_size;

	*argcp = 0;
	*argvp = NULL;

	if (str == NULL)
		return;

	// Scan past initial whitespace.
	while (*str != L'\0' && wcschr(whites, *str) != NULL) {
		if (*str == L'\\' && wcschr(whites, str[1]) != NULL) {
			str++;
		}
		str++;
	}
	if (*str == L'\0')
		return;

	argv = (WCHAR **) alloc_byte(sizeof(WCHAR *) * 8);
	if (argv == NULL)
		return;
	argv_size = 8;
	argc = 0;

	do {
		if (argc >= (argv_size - 1)) {
			argv_size += 8;
			argv = (WCHAR **) realloc((WCHAR **) argv,
								(unsigned) argv_size * sizeof(WCHAR *));
			if (argv == NULL)
				return;
		}

		argv[argc++] = str;

		while (*str != L'\0' && wcschr(whites, *str) == NULL) {
			if (*str == L'\\' && wcschr(whites, str[1]) != NULL) {
				WCHAR	*p;

				// What a hack. Copy the rest of the string
				// down by one byte to remove the backslash.
				// Don't forget to copy the null byte.
				for (p = str + 1; (*(p-1) = *p) != L'\0'; p++)
					;
			}
			str++;
		}
		while (*str != L'\0' && wcschr(whites, *str) != NULL)
			*str++ = L'\0';

	} while (*str != L'\0');

	argv[argc] = NULL;

	*argvp = argv;
	*argcp = argc;
}
