// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)fileio.c 	2.1 (Chris & John Downey) 7/29/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	fileio.c
* module function:
	File i/o routines.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

// Definition of a text file format.
//
// This structure may need additional entries to cope with very strange file
// formats (such as VMS).
struct tfformat
{
	int	tf_eolnchars[2];	// end of line markers 
	int	tf_eofchar; 		// end of file marker 
	int	tf_dynamic; 		// autodetect format? 
};

// Names of values for the P_format enumerated parameter.
//
// It is essential that these are in the same order as the fmt_...
// symbolic constants defined in xvi.h.
WCHAR	*fmt_strings[] = {
		L"cstring",
		L"macintosh",
		L"dos",
		L"qnx",
		L"tos",
		L"unix",
		NULL,
};

// Format structures.
//
// It is essential that these are in the same order as the fmt_...
// symbolic constants defined in xvi.h.
//
// We don't use '\r' or '\n' to define the end-of-line characters
// because some compilers interpret them differently & this code has
// to work the same on all systems.
//
// 'EOF's in this table means 'no character'.
static const struct tfformat tftable [] = {
	{ { L'\0',	    EOF			}, EOF, 	  FALSE	},	// fmt_CSTRING 
	{ { CTRL(L'M'), EOF			}, EOF, 	  FALSE	},	// fmt_MACINTOSH 
	{ { CTRL(L'M'), CTRL(L'J')	}, CTRL(L'Z'),TRUE	},	// fmt_DOS 
	{ { L'\036',    EOF			}, EOF, 	  FALSE	},	// fmt_QNX 
	{ { CTRL(L'M'), CTRL(L'J')	}, EOF, 	  TRUE	},	// fmt_TOS 
	{ { CTRL(L'J'), EOF			}, EOF, 	  FALSE	},	// fmt_UNIX 
};

// Index of last entry in tftable.
#define TFMAX	(sizeof tftable / sizeof (struct tfformat) - 1)

// Current text file format.
static struct tfformat curfmt = { { 0, 0 }, 0, FALSE };

#define eolnchars		curfmt.tf_eolnchars
#define eofchar 		curfmt.tf_eofchar

// Name of current text file format.
WCHAR *fmtname = L"INTERNAL ERROR";

// Definition of a text code type.
//
// Index of last valid entry in codetype_strings[]
#define CTMAX	3

// Names of values for the P_codetype enumerated parameter.
//
// It is essential that these are in the same order as the CODETYPE_...
// symbolic constants defined in win32.h.
WCHAR	*codetype_strings[CTMAX+1] = {
		L"multibyte",
		L"unicode",
		L"binary",
		NULL,
};

// Current text file code type.
static CODETYPE curct;

// Name of current text file code type.
WCHAR *ctname = L"INTERNAL ERROR";

// Copy the tftable entry indexed by tfindex into curfmt & update
// fmtname. Return FALSE if the parameter is invalid, otherwise TRUE.
//
// This is called from set_format() (below).
//
// Note that we copy a whole tfformat structure here, instead of just copying
// a pointer. This is so that curfmt.eolnchars & curfmt.eofchar will compile
// to absolute address references instead of indirections, which should be
// significantly more efficient because they are referenced for every
// character we read or write.
static BOOL txtformset(int tfindex)
{
	if (tfindex < 0 || tfindex > TFMAX)
		return FALSE;
	(void) memcpy((char *) &curfmt, (const char *) &tftable[tfindex],
												 sizeof curfmt);
	fmtname = fmt_strings[tfindex];
	return TRUE;
}

// Check value of P_format parameter.
BOOL set_format(Xviwin *window, Paramval new_value, BOOL interactive)
{
	if (!txtformset(new_value.pv_i)) {
		if (interactive) {
			show_error(window, L"Invalid text file format (%d)",
												new_value.pv_i);
		}
		return FALSE;
	}
	return TRUE;
}

static BOOL ctset(int ctindex)
{
	if (ctindex < 0 || ctindex > CTMAX) return FALSE;
	curct = (CODETYPE) ctindex;
	ctname = codetype_strings[ctindex];
	return TRUE;
}

// Check value of P_codetype parameter.
BOOL set_codetype(Xviwin *window, Paramval new_value, BOOL interactive)
{
	if (!ctset(new_value.pv_i)) {
		if (interactive) {
			show_error(window, L"Invalid text code type (%d)",
												new_value.pv_i);
		}
		return FALSE;
	}
	return TRUE;
}

// Find out if there's a format we know about with the single specified
// end-of-line character. If so, change to it.
static BOOL eolnhack(int c, BOOL fChangeFileInfo)
{
	int		tfindex;

	for (tfindex = 0; tfindex <= TFMAX; tfindex++) {
		const int		*eolp;

		eolp = tftable[tfindex].tf_eolnchars;
		if (eolp[0] == c && eolp[1] == EOF) {
			if (fChangeFileInfo) {
				txtformset(tfindex);
				set_param(P_format, tfindex, (WCHAR **) NULL);
				P_setchanged(P_format);
			}
			return TRUE;
		}
	}
	return FALSE;
}

// Read in the given file, filling in the given "head" and "tail"
// arguments with pointers to the first and last elements of the
// linked list of Lines; if nothing was read, both pointers are set to
// NULL. The return value is the number of lines read, if successful
// (this can be 0 for an empty file), or an error return code, which
// can be gf_NEWFILE, gf_CANTOPEN, gf_IOERR or gf_NOMEM.
//
// If there is an error, such as not being able to read the file or
// running out of memory, an error message is printed; otherwise, a
// statistics line is printed using show_message().
//
// The "extra_str" string is printed just after the filename in the
// displayed line, and is typically used for "Read Only" messages. If
// the file doesn't appear to exist, the filename is printed again,
// immediately followed by the "no_file_str" string, & we return
// gf_NEWFILE.
long get_file(Xviwin *window, WCHAR *filename, Line **headp, Line **tailp, WCHAR *extra_str, WCHAR *no_file_str, BOOL fChangeFileInfo)
{
	LPFILESTREAM		lpfs = NULL;	// ptr to open file 
	unsigned long		nchars; 		// number of chars read 
	unsigned long		nlines; 		// number of lines read 
	unsigned long		nulls;			// number of null chars 
	BOOL				incomplete; 	// incomplete last line 
	Line				*lptr = NULL;	// pointer to list of lines 
	Line				*last = NULL;	// last complete line
										// read in
	Line				*lp;			// line currently
										// being read in
	enum {
		at_soln,
		in_line,
		got_eolnc0,
		at_eoln,
		at_eof
	}					state;
	WCHAR				*buff = NULL;	// tmporary line buffer
										// text of line
										// being read in
	unsigned 			buffsize;		// current size of 'buff'
#	define	BUFFSIZE_UNIT 1024			// tmporary line buffer is
										// alloated as this size, and
										// lengthened by the size.
	int					col;			// current column in line 
	long				errcode;		// error code for error return

	// Allocate temporary line buffer.
	buffsize = BUFFSIZE_UNIT;
	buff = alloc_wchar(buffsize);
	if (buff == NULL) {
		errcode = gf_NOMEM;
		goto error_return;
	}

	// Load default file information.
	txtformset(Pen(P_format));
	ctset(Pen(P_codetype));

	// Try to open the file.
	lpfs = FileOpen(filename, L"r", curct);
	if (lpfs == NULL) {
		*headp = *tailp = NULL;
		if (FileExists(filename)) {
			show_error(window, L"Can't read \"%s\"", filename);
			errcode = gf_CANTOPEN;
			goto error_return;
		} else {
			show_message(window, L"\"%s\"%s [%s,%s]", filename, no_file_str, ctname, fmtname);
			errcode = gf_NEWFILE;
			goto error_return;
		}
	}

	// Autodetect file format
	if (lpfs->ct != curct) {
		ctset(lpfs->ct);
		if (fChangeFileInfo) {
			set_param(P_codetype, lpfs->ct, (WCHAR **) NULL);
			P_setchanged(P_codetype);
		}
	}

	show_message(window, L"\"%s\"[%s] %s", filename, ctname, extra_str);

	nchars = nlines = nulls = 0;
	col = 0;
	incomplete = FALSE;
	state = at_soln;
	while (state != at_eof) {

		int	c;

		c = FileGetc(lpfs);

		if (c == EOF || c == eofchar) {
			if (state != at_soln) {
				// Reached EOF in the middle of a line; what
				// we do here is to pretend we got a properly
				// terminated line, and assume that a
				// subsequent getc will still return EOF.
				incomplete = TRUE;
				state = at_eoln;
			} else {
				state = at_eof;
				break;
			}
		} else {
			nchars++;

			switch (state) {
			case at_soln:
				// We're at the start of a line.
				// no break
			case in_line:
				if (c == eolnchars[0]) {
					if (eolnchars[1] == EOF) {
						state = at_eoln;
					} else {
						state = got_eolnc0;
						continue;
					}
				} else if (c == eolnchars [1] && curfmt.tf_dynamic &&
									eolnhack(c, fChangeFileInfo)) {
					// If we get the second end-of-line
					// marker, but not the first, see if
					// we can accept the second one by
					// itself as an end-of-line.
					state = at_eoln;
				}
				break;
			case got_eolnc0:
				if (c == eolnchars[1]) {
					state = at_eoln;
				} else if (curfmt.tf_dynamic &&
						eolnhack(eolnchars[0], fChangeFileInfo)) {
					// If we get the first end-of-line
					// marker, but not the second, see
					// if we can accept the first one
					// by itself as an end-of-line.
					FileUngetc(c, lpfs);
					state = at_eoln;
				} else {
					// We can't. Just take the first one
					// literally.
					state = in_line;
					FileUngetc(c, lpfs);
					c = eolnchars [0];
				}
			}
		}

		if (col >= buffsize - 1) {
			// Line buffer for current line overflows. We adujust the size of it.
			buffsize += BUFFSIZE_UNIT;
			buff = (WCHAR*)realloc(buff, ((unsigned) buffsize) * sizeof(WCHAR));
			// If this fails, we squeak at the user and
			// then throw away the lines read in so far.
			if (buff == NULL) {
				if (lptr != NULL)
					throw_line(lptr);
				*headp = *tailp = NULL;
				errcode = gf_NOMEM;
				goto error_return;
			}
		}

		if (fChangeFileInfo && state == at_eoln) {
			// We've just read the first line. Only the first line is hacked
			// for file format analysis. Therefore 'fChangeFileInfo' should be
			// FALSE after this point.
			fChangeFileInfo = FALSE;
		}

		if (state == at_eoln) {
			// First null-terminate the current line.
			buff[col] = L'\0';

			// We're just read a line, and
			// we've got at least one character,
			// so we have to allocate a new Line
			// structure.
			//
			// If we can't do it, we throw away
			// the lines we've read in so far, &
			// return gf_NOMEM.
			if ((lp = newline(col + 1)) == NULL) {
				if (lptr != NULL) {
					throw_line(lptr);
				}
				*headp = *tailp = NULL;
				errcode = gf_NOMEM;
				goto error_return;
			}

			// We move data from current line buffer into Line structure.
			wcscpy(lp->l_text, buff);

			// Tack the line onto the end of the list,
			// and then point "last" at it.
			if (lptr == NULL) {
				lptr = lp;
				last = lptr;
			} else {
				last->l_next = lp;
				lp->l_prev = last;
				last = lp;
			}

			nlines++;
			col = 0;
			state = at_soln;
		} else {
			// Nulls are special; they can't show up in the file.
			if (c == L'\0') {
				nulls++;
				continue;
			}
			state = in_line;
			buff[col++] = c;
		}
	}

	free (buff);
	FileClose(lpfs);

	{
		// Assemble error messages for status line.
		Flexbuf 		errbuf;
		WCHAR			*errs;

		flexnew(&errbuf);
		if (nulls > 0) {
			(void) lformat(&errbuf, L" (%ld null character%s)",
					  nulls, (nulls == 1 ? L"" : L"s"));
		}
		if (incomplete) {
			(void) lformat(&errbuf, L" (incomplete last line)");
		}

		// Load default file information one more time to update 'ctname',
		// 'fmtname', etc.
		txtformset(Pen(P_format));
		ctset(Pen(P_codetype));

		// Show status line.
		errs = flexgetstr(&errbuf);
		show_message(window, L"\"%s\" [%s,%s]%s %ld/%ld%s",
								filename, ctname, fmtname, extra_str,
								nlines, nchars, errs);
		flexdelete(&errbuf);
	}

	*headp = lptr;
	*tailp = last;

	return nlines;

error_return:

	// clean up stuffs
	if (buff != NULL) free(buff);
	if (lpfs != NULL) FileClose(lpfs);

	return errcode;
}

// write_it - write to file 'fname' lines 'start' through 'end'
//
// If either 'start' or 'end' are NULL, the default
// is to use the start or end of the file respectively.
//
// Unless the "force" argument is TRUE, we do not write
// out buffers which have the "readonly" flag set.
BOOL write_it(Xviwin *window, WCHAR *fname, Line *start, Line *end, BOOL force)
{
	LPFILESTREAM		lpfs;
	unsigned long		nc;
	unsigned long		nl;
	Buffer				*buffer;

	// Load default file information.
	txtformset(Pen(P_format));
	ctset(Pen(P_codetype));

	buffer = window->w_buffer;

	if (is_readonly(buffer) && !force) {
		show_error(window, L"\"%s\" File is read only", fname);
		return FALSE;
	}

	show_message(window, L"\"%s\" [%s,%s]", fname, ctname, fmtname);

	// Preserve the buffer here so if the write fails it will at
	// least have been saved.
	if (!preservebuf(window)) {
		return FALSE;
	}

	if (!CanWriteOnFile(fname)) {
		show_error(window, L"\"%s\" Permission denied", fname);
		return FALSE;
	}

	lpfs = FileOpen(fname, L"w", curct);
	if (lpfs == NULL) {
		show_error(window, L"Can't write \"%s\"", fname);
		return FALSE;
	}

	if (put_file(window, lpfs, start, end, &nc, &nl, Pen(P_format)) == FALSE) {
		return FALSE;
	}

	show_message(window, L"\"%s\" [%s,%s] %ld/%ld", fname, ctname, fmtname, nl, nc);

	// Make sure any preserve file is removed if it isn't wanted.
	// It's not worth checking for the file's existence before
	// trying to remove it; the remove() will do the check anyway.
	if (Pn(P_preserve) < psv_PARANOID) {
		if (buffer->b_tempfname != NULL) {
			DeleteFile(buffer->b_tempfname);
		}
	}

	// If no start and end lines were specified, or they
	// were specified as the start and end of the buffer,
	// and we wrote out the whole file, then we can clear
	// the modified status. This must be safe.
	if ((start == NULL || start == buffer->b_file) &&
					(end == NULL || end == buffer->b_lastline->l_prev)) {
		buffer->b_flags &= ~FL_MODIFIED;
	}

	// Load default file information one more time to update 'ctname',
	// 'fmtname', etc.
	txtformset(Pen(P_format));
	ctset(Pen(P_codetype));

	return TRUE;
}

// Write out the buffer between the given two line pointers
// (which default to start and end of buffer) to the given file
// pointer. The reference parameters ncp and nlp are filled in
// with the number of characters and lines written to the file.
// The return value is TRUE for success, FALSE for all kinds of
// failure.
BOOL put_file(Xviwin *window, LPFILESTREAM lpfs, Line *start, Line *end, unsigned long *ncp, unsigned long *nlp, int fmt)
{
	Line				*lp;
	unsigned long		nchars;
	unsigned long		nlines;
	Buffer				*buffer;

	buffer = window->w_buffer;

	// If we were given a bound, start there. Otherwise just
	// start at the beginning of the file.
	if (start == NULL) {
		lp = buffer->b_file;
	} else {
		lp = start;
	}

	nlines = 0;
	nchars = 0;
	for ( ; lp != buffer->b_lastline; lp = lp->l_next) {

		WCHAR	*cp;

		// Write out the characters which comprise the line.
		// Register declarations are used for all variables
		// which form a part of this loop, in order to make
		// it as fast as possible.
		for (cp = lp->l_text; *cp != L'\0'; cp++) {
			if (FilePutc(*cp, lpfs) == EOF) {
				FileClose(lpfs);
				return FALSE;
			}
			nchars++;
		}

		if (FilePutc(tftable[fmt].tf_eolnchars[0], lpfs) == EOF) {
			FileClose(lpfs);
			return FALSE;
		}
		nchars++;

		if (tftable[fmt].tf_eolnchars[1] != EOF) {
			if (FilePutc(tftable[fmt].tf_eolnchars[1], lpfs) == EOF) {
				FileClose(lpfs);
				return FALSE;
			}
			nchars++;
		}

		nlines++;

		// If we were given an upper bound, and we
		// just did that line, then bag it now.
		if (end != NULL) {
			if (end == lp)
				break;
		}
	}

	if (FileClose(lpfs) != 0) {
		return FALSE;
	}

	// Success!
	if (ncp != NULL)
		*ncp = nchars;
	if (nlp != NULL)
		*nlp = nlines;
	return TRUE;
}

// Returns system default first end of line character
int system_eol1(void)
{
	return tftable[SYSTEM_TFF].tf_eolnchars[0];
}

// Returns system default second end of line character
int system_eol2(void)
{
	return tftable[SYSTEM_TFF].tf_eolnchars[1];
}

