// Copyright (c) 1990,1991,1992 Chris and John Downey 
#ifdef SCCID
static char *sccsid = "@(#)ex_cmds2.c	2.2 (Chris & John Downey) 8/3/92";
#endif

/***

* program name:
	wvi
* function:
	PD version of UNIX "vi" editor for WIN32, with extensions.
* module name:
	ex_cmds2.c
* module function:
	Command functions for miscellaneous ex (colon) commands.
	See ex_cmds1.c for file- and buffer-related colon commands.
* history:
	STEVIE - ST Editor for VI Enthusiasts, Version 3.10
	Originally by Tim Thompson (twitch!tjt)
	Extensive modifications by Tony Andrews (onecom!wldrdg!tony)
	Heavily modified by Chris & John Downey
	modified for WIN32 / UNICODE / C++ by K.Yoshizawa
		(PAF02413.niftyserve.or.jp)

***/

#include "xvi.h"

// Run shell command.
//
// You might think it would be easier to do this as
//
//		sys_endv();        (= ConsoleNormalMode())
//		system(command);   (= CallSystem())
//		sys_startv();      (= ConsoleViMode())
//		prompt(L"[Hit return to continue] ");
//		...
//
// but the trouble is that, with some systems, sys_startv() can be
// used to swap display pages, which would mean that the user would
// never have time to see the output from the command. (This applies
// to some terminals, as well as machines with memory-mappped video;
// cmdtool windows on Sun workstations also do the same thing.)
//
// This means we have to display the prompt before calling
// sys_startv(), so we just use good old fputs(). (We're trying not to
// use printf()).
void do_shcmd(Xviwin *window, WCHAR *command)
{
	WCHAR		*sh = NULL;
	int c;
	static WCHAR szHitMsg[] = L"[Hit return to continue] ";
	WCHAR		*cmdline;

	sh = Ps(P_shell);
	if (sh == NULL) {
		show_error(window, L"SHELL variable not set");
		return;
	}

	// +4 for L" /C ", +1 for L"\0".
	cmdline = alloc_wchar(wcslen(sh) + 4 + wcslen(command) + 1);
	wsprintf(cmdline, L"%s /C %s", sh, command);

	ConsolePrintf(L"\n");
	ConsoleFlushOutput();
	ConsoleNormalMode();

	CallSystem(cmdline);

	ConsolePrintf(L"%s", szHitMsg);
	ConsoleFlushOutput();

	while ((c = ConsoleGetc()) != L'\n' && c != L'\r' && c != EOF)
		;

	ConsoleViMode();

	free(cmdline);

	redraw_screen();
}

void do_shell(Xviwin *window)
{
	WCHAR		*sh = NULL;
	int sysret;

	sh = Ps(P_shell);
	if (sh == NULL) {
		show_error(window, L"SHELL variable not set");
		return;
	}

	FlushFileBuffers(hStdOut);
	ConsoleNormalMode();

	sysret = CallSystem(sh);

	ConsoleViMode();
	redraw_screen();

	if (sysret != 0) {
#ifdef STRERROR_AVAIL
		show_error(window, L"Can't execute %s [%s]", sh,
						(errno > 0 ? strerror(errno) : L"Unknown Error"));
#else	// strerror() not available 
		show_error(window, L"Can't execute %s", sh);
#endif	// strerror() not available 
	}
}

void do_suspend(Xviwin *window)
{
	if (State == NORMAL) {
		FlushFileBuffers(hStdOut);
		ConsoleNormalMode();

		SystemSuspend();

		ConsoleViMode();
		redraw_screen();
	}
}

void do_equals(Xviwin *window, Line *line)
{
	if (line == NULL) {
		// Default position is ".".
		line = window->w_cursor->p_line;
	}

	show_message(window, L"Line %ld", lineno(window->w_buffer, line));
}

void do_help(Xviwin *window)
{
	unsigned	savecho;

	savecho = echo;
	echo &= ~(e_SCROLL | e_REPORT | e_SHOWINFO);
	if (Ps(P_helpfile) != NULL && do_buffer(window, Ps(P_helpfile))) {
		// We use "curbuf" here because the new buffer will
		// have been made the current one by do_buffer().
		curbuf->b_flags |= FL_READONLY | FL_NOEDIT;
		move_window_to_cursor(curwin);
		show_file_info(curwin);
		update_window(curwin);
	}
	echo = savecho;
}

BOOL do_source(BOOL interactive, WCHAR *file)
{
	static WCHAR	cmdbuf[256];
	LPFILESTREAM	fs;
	int			c;
	WCHAR		*bufp;
	BOOL		literal;

	fs = FileOpen(file, L"r", DEF_TCT);
	if (fs == NULL) {
		if (interactive) {
			show_error(curwin, L"Can't open \"%s\"", file);
		}
		return(FALSE);
	}

	bufp = cmdbuf;
	literal = FALSE;
	while ((c = FileGetc(fs)) != EOF) {
		if (c == system_eol2()) continue; // Ignore second EOL character.
		if (!literal && (c == CTRL(L'V') || c == system_eol1())) {
			if (c == CTRL(L'V')) {
				literal = TRUE;
			} else if (c == system_eol1()) {
				if (kbdintr) {
					imessage = TRUE;
				} else if (bufp > cmdbuf) {
					*bufp = L'\0';
					do_colon(cmdbuf, FALSE);
				}
				bufp = cmdbuf;
			}
		} else {
			literal = FALSE;
			if (bufp < &cmdbuf[sizeof(cmdbuf) - 1]) {
				*bufp++ = c;
			}
		}
	}
	FileClose(fs);
	return TRUE;
}

// Change directory.
//
// With a NULL argument, change to the directory given by the HOME
// environment variable if it is defined; with an argument of "-",
// change back to the previous directory (like ksh).
//
// Return NULL pointer if OK, otherwise error message to say
// what went wrong.
WCHAR *do_chdir(WCHAR *dir)
{
#if 0
	static WCHAR *homedir = NULL;
	static WCHAR *prevdir = NULL;
	WCHAR		*ret;
	WCHAR		*curdir;

	if (dir == NULL && homedir == NULL &&
								(homedir = getenv(L"HOME")) == NULL) {
		return(L"HOME environment variable not set");
	}

	if (dir == NULL) {
		dir = homedir;
	} else if (*dir == L'-' && dir[1] == L'\0') {
		if (prevdir == NULL) {
			return(L"No previous directory");
		} else {
			dir = prevdir;
		}
	}

	curdir = (WCHAR*)WMALLOC(MAXPATHLEN + 2);
	if (curdir != NULL && getcwd(curdir, MAXPATHLEN + 2) == NULL) {
		free(curdir);
		curdir = NULL;
	}
	ret = (chdir(dir) == 0 ? NULL : L"Invalid directory");
	if (prevdir) {
		free(prevdir);
		prevdir = NULL;
	}
	if (curdir) {
		prevdir = (WCHAR*)WREALLOC(curdir, (unsigned) wcslen(curdir) + 1);
	}
	return(ret);
#endif
return NULL;
}

void do_cdmy(int type, Line *l1, Line *l2, Line *destline)
{
	Posn		p1, p2;
	Posn		destpos;

	p1.p_line = (l1 != NULL) ? l1 : curwin->w_cursor->p_line;
	p2.p_line = (l2 != NULL) ? l2 : p1.p_line;
	p1.p_index = p2.p_index = 0;

	if (type == L'c' || type == L'm') {
		if (destline == NULL) {
			show_error(curwin, L"No destination specified");
			return;
		}
	}

	// Check that the destination is not inside
	// the source range for "move" operations.
	if (type == L'm') {
		unsigned long	destlineno;

		destlineno = lineno(curbuf, destline);
		if (destlineno >= lineno(curbuf, p1.p_line) &&
							destlineno <= lineno(curbuf, p2.p_line)) {
			show_error(curwin, L"Source conflicts with destination of move");
			return;
		}
	}

	// Yank the text to be copied.
	if (do_yank(curbuf, &p1, &p2, FALSE, L'@') == FALSE) {
		show_error(curwin, L"Not enough memory to yank text");
		return;
	}

	if (!start_command(curwin)) {
		return;
	}

	switch (type) {
	case L'd':					// delete 
	case L'm':					// move 
		move_cursor(curwin, p1.p_line, 0);
		repllines(curwin, p1.p_line, cntllines(p1.p_line, p2.p_line),
												(Line *) NULL);
		update_buffer(curbuf);
		cursupdate(curwin);
		begin_line(curwin, TRUE);
	}

	switch (type) {
	case L'c':					// copy 
	case L'm':					// move 
		// And put it back at the destination point.
		destpos.p_line = destline;
		destpos.p_index = 0;
		do_put(curwin, &destpos, FORWARD, L'@');

		update_buffer(curbuf);
		cursupdate(curwin);
	}

	end_command(curwin);
}
