/*
 * NQC interface
 * Copyright 1998-2000 (C) Eiichiroh Itoh
 *
 * Ported from original nqc's nqc/nqc.cpp
 *  The Initial Developer of this code is David Baum.
 *  Portions created by David Baum are Copyright (C) 1998 David Baum.
 *  All Rights Reserved.
 *
 * 1999/12/29:Eiichiroh Itoh
 *  (1) Initial development
 * Modification
 * 2000/02/23:Eiichiroh Itoh
 *  (1) Modify for nqc-2.1r1
 *
 */
#include	"cetools.h"
#include	"cenqc.h"
#include	"cedialog.h"
/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */
#ifdef	_WIN32
#include	"Fragment.h"
#include	"Function.h"
#include	"Expansion.h"
#endif	/* _WIN32 */
#include "Program.h"
#include "RCX_Image.h"
#include "RCX_Link.h"
#include "Symbol.h"
#include "PreProc.h"
#include "parser.h"
#include "Macro.h"
#include "RCX_Cmd.h"
#include "RCX_Log.h"
#include "SRecord.h"
#include "AutoFree.h"
#include "DirList.h"
#include "Buffer.h"
#include "Error.h"
#include "Compiler.h"
#include "RCX_Disasm.h"
#include "CmdLine.h"



// use this to check for memory leaks in the compiler
//#define CHECK_LEAKS

// use these to debug the LEXER
//#define TEST_LEXER	1	// test lexer only
//#define TEST_LEXER	2	// test lexer and pre-processor



#ifdef CHECK_LEAKS
#include <DebugNew.h>
#endif

// some win32 consoles are lame...use stdout instead of stderr for usage, etc.
#define STDERR stderr

FILE *gErrorStream = stderr;

#define VERSION	"2.1 r1"

#define kMaxPrintedErrors	10



class AutoLink : public RCX_Link
{
public:
				AutoLink() : fSerialPort(0), fOpen(false) {}
				~AutoLink()	{ Close(); }

	RCX_Result	Open();
	void		Close();
	RCX_Result	Send(const RCX_Cmd *cmd, bool retry=true);
	
	void		SetSerialPort(const char *sp)	{ fSerialPort = sp; }

	bool		DownloadProgress(int soFar, int total);

private:
	const char*	fSerialPort;
	bool		fOpen;		
};



class MyCompiler : public Compiler, public ErrorHandler
{
public:
	MyCompiler()	{ pfDirs = new DirList ; }
	~MyCompiler()	{ delete pfDirs ; }
				
	Buffer	*GetBuffer( const char *name ) ;

	void	AddError( const Error &e, const LexLocation *loc ) ;
	void	AddDir( const char *dirspec)	{ pfDirs->Add( dirspec ) ; }
	void	DeleteDir() { delete pfDirs ; pfDirs = new DirList ; }

private:
	DirList	*pfDirs ;
} gMyCompiler ;



#define kMaxFirmware 65536
#define kOldIncludePathEnv	"NQCC_INCLUDE"
#define kNewIncludePathEnv	"NQC_INCLUDE"
#define kOptionsEnv			"NQC_OPTIONS"
#define kLowBattery	6600

#define kRCXFileExtension TEXT(".rcx")
#define kNQCFileExtension TEXT(".nqc")



// error codes in addition to RCX_Result codes
#define kUsageError	(kRCX_LastError - 1)
#define kQuietError	(kRCX_LastError - 2)

// codes for the actions
enum
{
	kFirstActionCode = 256,
	kDatalogCode = kFirstActionCode,
	kDatalogFullCode,
	kClearMemoryCode,
	kFirmwareCode,
	kFirmware4xCode,
	kNearCode,
	kFarCode,
	kWatchCode,
	kSleepCode,
	kRunCode,
	kProgramCode,
	kMessageCode,
	kRawCode,
	kRemoteCode,
	kCompileStdinCode
};

// these must be in the same order as the codes for the long options
static const char *sActionNames[] = {
	"datalog", "datalog_full", "clear", "firmware", "firmfast", "near", "far", "watch",
	"sleep", "run", "pgm", "msg", "raw", "remote", ""
};



struct Request
{
	const char *fSourceFile;
	const char *fOutputFile;
	const char *fListFile;
	bool	fListing;
	bool	fDownload;
	int		fFlags;
};

#ifndef	_WIN32
static int GetActionCode(const char *arg);
static RCX_Result ProcessCommandLine(int argc, char **argv);
static void PrintError(RCX_Result error, const char *filename = 0);
#endif	/* _WIN32 */
static void PrintUsage();
#ifndef	_WIN32
static void DefineMacro(const char *text);
static RCX_Result ProcessFile(const char *sourceFile, const Request &req);
#endif	/* _WIN32 */
static char *CreateFilename(const char *source, const char *oldExt, const char *newExt);
static const char *LeafName(const char *filename);
static int CheckExtension(const char *s1, const char *ext);
static RCX_Image *Compile(const char *sourceFile,  int flags);
static bool GenerateListing(RCX_Image *image, const char *filename);
static RCX_Result Download(RCX_Image *image);
static RCX_Result UploadDatalog(bool verbose);
static RCX_Result DownloadFirmware(const char *filename, bool fast);
static RCX_Result SetWatch(const char *timeSpec);
static RCX_Result SetErrorFile(const char *filename);
static RCX_Result SendRawCommand(const char *text);
static RCX_Result ClearMemory();
static RCX_Result SetTarget(const char *name);
static RCX_Result SendRemote(const char *event, int repeat);
static void PrintToken(int t, TokenVal v);


AutoLink gLink;
RCX_TargetType gTargetType = kRCX_RCXTarget;


#ifndef	_WIN32
int main(int argc, char **argv)
{
	RCX_Result result;

	// add any default include paths
	gMyCompiler.AddDir(getenv(kOldIncludePathEnv));
	gMyCompiler.AddDir(getenv(kNewIncludePathEnv));
	
	result = ProcessCommandLine(argc, argv);
	gLink.Close();
	
	PrintError(result);

	if (gErrorStream != stderr &&
		gErrorStream != stdout)
		fclose(gErrorStream);
	
#ifdef DEBUG_NEW
	printf("%d allocations, %d total bytes, %d max\n", gDebugNewAllocCount, gDebugNewAllocCurr, gDebugNewAllocMax);
#endif
	
	
	return (result==kRCX_OK) ? 0 : -1;
}


RCX_Result ProcessCommandLine(int argc, char ** argv)
{
	bool optionsOK = true;
	bool fileProcessed = false;
	Request req = { 0, 0, 0, false, false, 0 };
	CmdLine args;
	RCX_Result result = kRCX_OK;
	RCX_Cmd cmd;

	// first add environment options
	args.Parse(getenv(kOptionsEnv));

	// add all command line args after the first one
#ifdef WIN32
	// For Win32, bypass the argc/argv array and get the raw command line
	// This is for backwards compatability with RcxCC which doesn't like its
	// args to be escaped by the runtime
	#pragma unused(argc, argv)
	args.Parse(GetCommandLineA(), 1);
#else
	if (argc > 1)
		args.Add(argc-1, argv+1);
#endif
	
	// process the args
	while(args.Remain() && !RCX_ERROR(result))
	{
		const char* a=args.Next();
		bool isOption = (a[0] == '-');
		
#ifdef WIN32
		if (a[0]=='/') isOption = true;
#endif

		if (isOption)
		{
			int code = GetActionCode(a+1);

			if (code)
			{
				optionsOK = false;
			}
			else
			{
				if (!optionsOK) return kUsageError;
				code = a[1];
			}
			
			switch(code)
			{
				// options
				case '1':
					req.fFlags |= Compiler::kCompat_Flag;
					break;
				case 'D':
					DefineMacro(a+2);
					break;
				case 'E':
					result = SetErrorFile(a+2);
					break;
				case 'I':
					gMyCompiler.AddDir(a+2);
					break;
				case 'L':
					req.fListing = true;
					req.fListFile = a[2] ? a+2 : 0;
					break;
				case 'O':
					req.fOutputFile = a+2;
					break;
				case 'S':
					gLink.SetSerialPort(a+2);
					break;
				case 'T':
					result = SetTarget(a+2);
					break;
				case 'U':
					Compiler::Get()->Undefine(a+2);
					break;
				case 'd':
					req.fDownload = true;
					break;
				case 'l':
					req.fListing = true;
					break;
				case 'n':
					req.fFlags |= Compiler::kNoSysFile_Flag;
					break;
				case 't':
					if (!args.Remain()) return kUsageError;
					gLink.SetRxTimeout(args.NextInt());
					break;
				case 'v':
					gLink.SetVerbose(true);
					break;

				// deprecated
				case 'c':
					gTargetType = kRCX_CMTarget;
					break;
				case 's':
					gTargetType = kRCX_ScoutTarget;
					break;

				// actions
				case kCompileStdinCode:
					result = ProcessFile(nil, req);
					fileProcessed = true;
					break;
				case kDatalogCode:
					result = UploadDatalog(false);
					break;
				case kDatalogFullCode:
					result = UploadDatalog(true);
					break;
				case kClearMemoryCode:
					result = ClearMemory();
					break;
				case kFirmwareCode:
					if (!args.Remain()) return kUsageError;
					result = DownloadFirmware(args.Next(), false);
					break;
				case kFirmware4xCode:
					if (!args.Remain()) return kUsageError;
					result = DownloadFirmware(args.Next(), true);
					break;
				case kNearCode:
					result = gLink.Send(cmd.Set(kRCX_IRModeOp, 0));
					break;
				case kFarCode:
					result = gLink.Send(cmd.Set(kRCX_IRModeOp, 1));
					break;
				case kWatchCode:
					if (!args.Remain()) return kUsageError;
					result = SetWatch(args.Next());
					break;
				case kSleepCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_AutoOffOp, (UByte)args.NextInt()));
					break;
				case kRunCode:
					result = gLink.Send(cmd.Set(kRCX_StartTaskOp, 0));
					break;
				case kProgramCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_SelectProgramOp, (UByte)(args.NextInt()-1)));
					break;
				case kMessageCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_Message, (UByte)(args.NextInt())), false);
					break;
				case kRawCode:
					if (!args.Remain()) return kUsageError;
					result = SendRawCommand(args.Next());					
					break;
				case kRemoteCode:
					if (args.Remain() < 2) return kUsageError;
					{
						const char *event = args.Next();
						int repeat = args.NextInt();
						result = SendRemote(event, repeat);
					}
					break;
				default:
					return kUsageError;
			}
		}
		else if (!fileProcessed)
		{
			result = ProcessFile(a, req);

#ifdef CHECK_LEAKS
			int firstAllocCount = gDebugNewAllocCount;
			Compiler::Get()->Reset();
			result = ProcessFile(a, req);
			printf("%d leaked allocations\n", gDebugNewAllocCount - firstAllocCount);
#endif
			optionsOK = false;
			fileProcessed = true;
		}
		else
			return kUsageError;
	}
	
	// check if we did anything (compile and/or actions)
	if (optionsOK) return kUsageError;
	
	return result;
}


RCX_Result SetTarget(const char *name)
{
	switch(name[0])
	{
		case 'r':
		case 'R':
			gTargetType = kRCX_RCXTarget;
			break;
		case 'c':
		case 'C':
			gTargetType = kRCX_CMTarget;
			break;
		case 's':
		case 'S':
			gTargetType = kRCX_ScoutTarget;
			break;
		default:
			return kUsageError;		
	}
	
	return kRCX_OK;
}


RCX_Result SetWatch(const char *timeSpec)
{
	int hour;
	int minute;
	RCX_Cmd cmd;
	
	if (strcmp(timeSpec, "now")==0)
	{
		time_t t;
		struct tm *tmp;
		
		time(&t);
		tmp = localtime(&t);
		hour = tmp->tm_hour;
		minute = tmp->tm_min;
	}
	else
	{
		int t = atoi(timeSpec);
		hour = t / 100;
		minute = t % 100;
	}
	
	return gLink.Send(cmd.Set(kRCX_SetWatchOp, (UByte)hour, (UByte)minute));
}


RCX_Result ProcessFile(const char *sourceFile, const Request &req)
{
	RCX_Image *image;
	RCX_Result result = kRCX_OK;
	bool ok = true;
	
	if (sourceFile && CheckExtension(sourceFile, kRCXFileExtension))
	{
		// load RCX image file
		image = new RCX_Image();
		result = image->Read(sourceFile);
		if (RCX_ERROR(result))
		{
			PrintError(result, sourceFile);
			delete image;
			return kQuietError;
		}
	}
	else
	{
		// compile file
		image = Compile(sourceFile, req.fFlags);
		
		if (!image)
		{
			int errors = ErrorHandler::Get()->GetCount();
			
			if (errors)
				fprintf(gErrorStream, "# %d error%s during compilation\n", errors, errors==1 ? "" : "s");
			return kQuietError;
		}
		
		const char *outputFile = req.fOutputFile;
		char *newFilename = 0;
		
		if (!req.fDownload && !req.fListing && !req.fOutputFile && sourceFile)
			outputFile = newFilename = CreateFilename(LeafName(sourceFile), kNQCFileExtension, kRCXFileExtension);
		
		if (outputFile)
		{
			if (!image->Write(outputFile))
			{
				fprintf(gErrorStream, "Error: could not create output file \"%s\"\n", outputFile);
				ok = false;
			}
		}
		
		if (newFilename)
			delete [] newFilename;
	}

	// generate the listing
	if (req.fListing)
	{
		if (!GenerateListing(image, req.fListFile))
			ok = false;
	}

	if (req.fDownload)
	{
		result = Download(image);
	}

	// Check for the case of a listing or output file failure but
	// download ok.  In this case an error code must still be returned
	// so that command line processing stops and main() indicates
	// the failure
	if (result==kRCX_OK && !ok)
		result = kQuietError;
		
	delete image;
	return result;
}


bool GenerateListing(RCX_Image *image, const char *filename)
{
	FILE *fp;
	
	if (filename)
	{
		fp = fopen(filename, "w");
		if (!fp)
		{
			fprintf(STDERR, "Error: could not generate listing to file %s\n", filename);
			return false;
		}
	}
	else
		fp = stdout;

	RCX_StdioPrinter dst(fp);
	image->Print(&dst);
	
	if (fp != stdout)
		fclose(fp);

	return true;	
}
#endif	/* _WIN32 */


RCX_Result Download(RCX_Image *image)
{
	RCX_Result result = kRCX_OK;
	RCX_Cmd cmd;
	
	fprintf(STDERR, "Downloading Program:");
	result = gLink.Open();
	if (result != kRCX_OK) goto ErrorReturn;
	
	result = image->Download(&gLink);
	if (result != kRCX_OK) goto ErrorReturn;	
	
	fprintf(STDERR, "complete\n");
	
	result = gLink.GetBatteryLevel();
	if (!RCX_ERROR(result))
	{
#ifdef	WINCE
		fprintf( STDERR, "Battery Level = %d V\n", result ) ;
#else	/* WINCE */
		fprintf(STDERR, "Battery Level = %3.1f V\n", (double)result / 1000);
#endif	/* WINCE */
		if (result < kLowBattery)
			fprintf(STDERR, "*** Warning: batteries are low ***\n");
	}
	
	return kRCX_OK;
	
ErrorReturn:
	fprintf(STDERR, "error\n");
	return result;
}


RCX_Image *Compile(const char *sourceFile, int flags)
{
	Buffer *mainBuf;

	if (sourceFile)
	{
		FILE *fp  = fopen(sourceFile, "rb");
		if (!fp)
		{
			fprintf(gErrorStream, "Error: could not open file \"%s\"\n", sourceFile);
			return nil;
		}
		
		mainBuf = new Buffer();
		mainBuf->Create(sourceFile, fp);
		fclose(fp);
	}
	else
	{
		mainBuf = new Buffer();
		mainBuf->Create("<stdin>", stdin);
	}

#ifdef TEST_LEXER
	LexPush(mainBuf);

	int t;
	TokenVal v;
#if TEST_LEXER > 1
	while((t=gPreProc->Get(v)) != 0)
#else
	while((t=yylex(v)) != 0)
#endif
	{
		LexLocation loc;
		LexCurrentLocation(loc);
		printf("%3d : ", loc.fLine);
		PrintToken(t, v);
	}
	
	return 0;
#else
	return Compiler::Get()->Compile(mainBuf, getTarget(gTargetType), flags);
#endif
}


RCX_Result UploadDatalog(bool verbose)
{
	RCX_Log log;
	RCX_Result result;
	int i;
	
	fprintf(STDERR, "Uploading Datalog");
	
	result = gLink.Open();
	if (RCX_ERROR(result)) return result;
	
	result = log.Upload(&gLink);
	if (RCX_ERROR(result)) return result;
	
	fprintf(STDERR, "\n");
	
	for(i=0; i<log.GetLength(); i++)
	{
		char line[256];
		log.SPrintEntry(line, i, verbose);
		printf("%s\n", line);
	}
	
	return kRCX_OK;
}


RCX_Result ClearMemory()
{
	RCX_Result result;
	RCX_Cmd cmd;
	
	// halt all tasks
	result = gLink.Send(cmd.Set(kRCX_StopAllOp));
	if (RCX_ERROR(result)) return result;
	
	for(UByte p=0; p<5; p++)
	{
		// select and delete program
		result = gLink.Send(cmd.Set(kRCX_SelectProgramOp, p));
		if (RCX_ERROR(result)) return result;
	
		result = gLink.Send(cmd.MakeDeleteSubs());
		if (RCX_ERROR(result)) return result;

		result = gLink.Send(cmd.MakeDeleteTasks());
		if (RCX_ERROR(result)) return result;
	}

	result = gLink.Send(cmd.Set(kRCX_SetDatalogOp, 0, 0));
	return result;
}


RCX_Result DownloadFirmware(const char *filename, bool fast)
{
	FILE *fp;
	SRecord srec;
	bool ok;
	RCX_Result result;
	ULong rom, ram;
	
	fp = fopen(filename, "r");
	if (!fp)
	{
		fprintf(STDERR, "Error: could not open file \'%s\'\n", filename);
		return kQuietError;
	}
	
	ok = srec.Read(fp, kMaxFirmware);
	fclose(fp);
	
	if (!ok)
	{
		fprintf(STDERR, "Error: \'%s\' is not a valid S-Record file\n", filename);
		return kQuietError;
	}

	result = gLink.Open();
	if (RCX_ERROR(result)) return result;
	
	fprintf(STDERR, "Downloading firmware:");
	fflush(STDERR);
	result = gLink.SendFirmware(srec.GetData(), srec.GetLength(), srec.GetStart(), fast);
	fputc('\n', STDERR);
	if (RCX_ERROR(result)) return result;
	
	result = gLink.GetVersion(rom, ram);
	if (RCX_ERROR(result)) return result;

	fprintf(STDERR, "Current Version: %08lx/%08lx\n", rom, ram);

	return result;
}


RCX_Result SendRawCommand(const char *text)
{
	int length = (int)strlen(text);
	RCX_Cmd cmd;
	RCX_Result result;
	
	// we need an even number of chars in text
	if (length & 1) return kUsageError;
	
	// determine actual length of command
	length /= 2;
	cmd.SetLength(length);
	
	for(int i=0; i<length; i++)
	{
		int byte = SRecord::ReadHexByte(text);
		if (byte == -1) return kUsageError;
		cmd[i] = (UByte) byte;
		text+=2;
	}
	
	result = gLink.Send(&cmd);

	if (result > 0)
	{
		for(int i=0; i<result; i++)
			printf("%02x ", gLink.GetReplyByte(i));
		printf("\n");
	}
	
	return result;
}


RCX_Result SendRemote(const char *event, int repeat)
{
	RCX_Cmd cmd;
	int low, high;
	
	if (repeat == 0) return kUsageError;
	
	if (strlen(event) != 4)
		return kUsageError;
	
	high = SRecord::ReadHexByte(event);
	low = SRecord::ReadHexByte(event+2);
	if (high==-1 || low==-1) return kUsageError;
	

	cmd.Set(kRCX_Remote, low, high);
	
	while(repeat--)
	{
		gLink.Send(&cmd, false);
	}
	
	return kRCX_OK;
}


char *CreateFilename(const char *source, const char *oldExt, const char *newExt)
{
	char *filename;
	size_t n = strlen(source);
	size_t newExtLen = strlen(newExt);

	if (CheckExtension(source, oldExt))
		n -= strlen(oldExt);

	filename = new char[n + newExtLen + 1];
	memcpy(filename, source, n);
	strcpy(filename + n, newExt);
	
	return filename;
}


int CheckExtension(const char *s1, const char *ext)
{
	size_t slen = strlen(s1);
	size_t plen = strlen(ext);
	unsigned i;
	
	if (slen < plen) return 0;
	
	for(i=0; i<plen; i++)
		if (tolower(s1[slen-plen+i]) != tolower(ext[i])) return 0;
	
	return 1;
}


const char *LeafName(const char *filename)
{
	const char *ptr;

	// start at end and work towards beginning
	ptr = filename + strlen(filename) - 1;

	// test each character (including first one)
	while(ptr >= filename)
	{
		// check for delimiter
		if (*ptr == DIR_DELIMITER) break;

#ifdef WIN32
		// extra check for ':' in paths for Windows
		if (*ptr == ':') break;
#endif

		--ptr;
	}
	
	// we either stopped at the char before the first, or at
	// the delimiter - either way return pointer to next char
	return ptr+1;
}


RCX_Result SetErrorFile(const char *filename)
{
	FILE *fp;
	
	if (*filename==0)
	{
		gErrorStream = stdout;
		return kRCX_OK;
	}
	
	fp = fopen(filename, "w");
	if (!fp)
	{
		fprintf(STDERR, "Error: could not open error file \'%s\'\n", filename);
		return kQuietError;
	}
	
	gErrorStream = fp;
	return kRCX_OK;
}


void DefineMacro(const char *text)
{
	const char *body;
	body = strchr(text, '=');
	if (body)
	{
		// create a copy of the symbol name
		int length = body - text;
		char *name = new char[length+1];
		memcpy(name, text, (size_t)length);
		name[length] = 0;
		
		Compiler::Get()->Define(name, body+1);
		
		delete [] name;
	}
	else	
	{
		Compiler::Get()->Define(text);
	}
}


#ifndef	_WIN32
int GetActionCode(const char *arg)
{
	for(int i=0; i< (int)(sizeof(sActionNames)/sizeof(const char *)); ++i)
		if (strcmp(sActionNames[i], arg)==0) return i+kFirstActionCode;

	return 0;
}
#endif	/* _WIN32 */


static
void PrintError(RCX_Result error, const char *filename)
{
	const char *targetName = getTarget(gTargetType)->fName;
	if (!filename) filename = "?";
	
	if (error >= 0) return;
	
	switch(error)
	{
		case kRCX_OpenSerialError:
			fprintf(STDERR, "Could not open serial port\n");
			break;
		case kRCX_IREchoError:
			fprintf(STDERR, "Problem talking to IR device\n");
			break;
		case kRCX_ReplyError:
			fprintf(STDERR, "No reply from %s\n", targetName);
			break;
		case kRCX_MemFullError:
			fprintf(STDERR, "Not enough free memory in %s to download program\n", targetName);
			break;
		case kRCX_FileError:
			fprintf(STDERR, "Could not access file \'%s\'\n", filename);
			break;
		case kRCX_FormatError:
			fprintf(STDERR, "File \'%s\' is not a valid RCX image\n", filename);
			break;
#ifndef	_WIN32
		case kUsageError:
			PrintUsage();
			break;
#endif	/* _WIN32 */
		case kQuietError:
			break;
		default:
			fprintf(STDERR, "Error #%d\n", -error);
			break;
	}
}


#ifdef	_WIN32
#ifdef	UNICODE
static void
PrintError( RCX_Result error, LPCTSTR filename )
{
	char	szSjis[ 256 ] ;

	Wcstombs( szSjis, filename, 256 ) ;
	PrintError( error, szSjis ) ;
}
#endif	/* UNICODE */

void
CE_print_copyright( HWND hwndError )
{
	AttachWinToStream( CE_stderr, hwndError ) ;
	Fprintf( CE_stderr, "CENQC version %d.%d (built "__DATE__", " __TIME__")\n",
			 g_dwAppVersion / 100, g_dwAppVersion % 100 ) ;
	Fprintf( CE_stderr, "  Copyright (C) 1999, 2000 Eiichiro Ito.  All Rights Reserved.\n" ) ;
	Fprintf( CE_stderr," based on nqc version "VERSION" \n" ) ;
	Fprintf( CE_stderr,"   Copyright (C) 1998-2000 David Baum.  All Rights Reserved.\n" ) ;
}
#else	/* _WIN32 */
void PrintUsage()
{
	fprintf(STDERR,"nqc version "VERSION" (built "__DATE__", " __TIME__")\n");
	fprintf(STDERR,"     Copyright (C) 1998-2000 David Baum.  All Rights Reserved.\n");
	fprintf(STDERR,"Usage: nqc [options] [actions] [ - | filename ] [actions]\n");
	fprintf(STDERR,"   - : read from stdin instead of a source_file\n");	
	fprintf(STDERR,"Options:\n");	
	fprintf(STDERR,"   -1: use NQC 1.x compatability mode\n");	
	fprintf(STDERR,"   -T<target>: target can be RCX, CM or Scout\n");	
	fprintf(STDERR,"   -d: download program\n");
	fprintf(STDERR,"   -n: prevent the system file (rcx.nqh) from being included\n");
	fprintf(STDERR,"   -D<sym>[=<value>] : define macro <sym>\n");
	fprintf(STDERR,"   -E[<filename>] : write compiler errors to <filename> (or stdout)\n");
	fprintf(STDERR,"   -I<path>: search <path> for include files\n");
	fprintf(STDERR,"   -L[<filename>] : generate code listing to <filename> (or stdout)\n");
	fprintf(STDERR,"   -O<outfile>: specify output file\n");
	fprintf(STDERR,"   -S<portname>: specify serial port\n");
	fprintf(STDERR,"   -U<sym>: undefine macro <sym>\n");
	fprintf(STDERR,"Actions:\n");	
	fprintf(STDERR,"   -run: run current program\n");
	fprintf(STDERR,"   -pgm <number>: select program number\n");
	fprintf(STDERR,"   -datalog | -datalog_full: upload datalog\n");
	fprintf(STDERR,"   -near : set IR to near mode\n");
	fprintf(STDERR,"   -far : set IR to far mode\n");
	fprintf(STDERR,"   -watch <time> | now : set RCX time\n");
	fprintf(STDERR,"   -firmware <filename> : download firmware\n");
	fprintf(STDERR,"   -firmfast <filename> : download firmware at quad speed\n");
	fprintf(STDERR,"   -sleep <timeout> : set RCX sleep timeout\n");
	fprintf(STDERR,"   -msg <number> : send IR message to RCX\n");
	fprintf(STDERR,"   -raw <data> : format data as a packet and send to RCX\n");
	fprintf(STDERR,"   -remote <value> <repeat> : send a remote command to the RCX\n");
	fprintf(STDERR,"   -clear : erase all programs and datalog in RCX\n");
}
#endif	/* _WIN32 */


RCX_Result AutoLink::Open()
{
	RCX_Result result;
	
	if (!fOpen)
	{
		SetTarget(gTargetType);
		result = RCX_Link::Open(fSerialPort);
		if (RCX_ERROR(result)) return result;
		
		fOpen = true;
	}
	return kRCX_OK;
}


void AutoLink::Close()
{
	if (fOpen)
	{
		RCX_Link::Close();
		fOpen = false;
	}
}


RCX_Result AutoLink::Send(const RCX_Cmd *cmd, bool retry)
{
	RCX_Result result;
	
	result = Open();
	if (RCX_ERROR(result)) return result;
	
	if (retry)
	{
		result = Sync();
		if (RCX_ERROR(result)) return result;
	}
	
	result = RCX_Link::Send(cmd, retry);

	return retry ? result : kRCX_OK;
}


bool AutoLink::DownloadProgress(int /* soFar */, int /* total */)
{
	fputc('.', STDERR);
	fflush(STDERR);
	return true;
}



void MyCompiler::AddError(const Error &e, const LexLocation *loc)
{
	char msg[Error::kMaxErrorMsg];
	e.SPrint(msg);

	if (e.IsWarning())
	{
		fprintf(gErrorStream, "# Warning: ");
	}
	else
	{
		int errorCount = ErrorHandler::Get()->GetCount();
		// only print the first few errors
		if (errorCount > kMaxPrintedErrors)
		{
			if (errorCount == kMaxPrintedErrors+1)
				fprintf(gErrorStream, "Too many errors - only first %d reported\n", kMaxPrintedErrors);
			return;
		}

		fprintf(gErrorStream, "# Error: ");
	}

	fprintf(gErrorStream, "%s\n", msg);

	if (loc)
	{
		// get line number (and adjust for NL errors)
		int line = loc->fLine;
		if (loc->fCol == -1)
			line--;
	
		fprintf(gErrorStream, "File \"%s\" ; line %d\n", loc->fBuffer->GetName(), line);
	
		if ((loc->fCol != -1))
		{
			int i;
			const char *ptr = loc->fBuffer->GetLine(line);
			if (ptr)
			{
				fprintf(gErrorStream, "# ");
				
				for(; *ptr != '\n'; ptr++)
				{
					putc((*ptr == '\t') ? ' ' : *ptr, gErrorStream);
				}
				
				fprintf(gErrorStream,"\n# ");
				for(i=0; i<loc->fCol; i++)
					putc(' ', gErrorStream);
				for(i=0; i<loc->fLength; i++)
					putc('^', gErrorStream);
				fprintf(gErrorStream,"\n");
			}
		}
	}

	fprintf(gErrorStream, "#----------------------------------------------------------\n");
}


Buffer *MyCompiler::GetBuffer(const char *name)
{
	static char pathname[DirList::kMaxPathname];	

#ifdef	_WIN32
	if (!pfDirs->Find(name, pathname))
#else	/* _WIN32 */
	if (!fDirs.Find(name, pathname))
#endif	/* _WIN32 */
		return nil;

	Buffer *buf = new Buffer();
	
	if (buf->Create(name, pathname))
		return buf;
	else
	{
		delete buf;
		return 0;
	}
}


void PrintToken(int t, TokenVal v)
{
	printf("%3d / ", t);
	if (t < 256)
		printf("%c", t);
	else switch(t)
	{
		case ID:
			printf("%s",v.fSymbol->GetKey());
			break;
		case NUMBER:
			printf("%d", v.fInt);
			break;
		case IF:
			printf("if");
			break;
		case STRING:
			printf("%s", v.fString);
			break;
		default:
			printf("0x%08x", v.fInt);
			break;
	}
	
	printf("\n");
}


int			g_fFlags = 0 ;
char		g_sjisSerialPort[ MAX_PATH ] = "COM1:" ;
RCX_Image	*g_image = NULL ;

static void		SetupCompilerOption() ;
static void		SetupDownloadOption() ;

BOOL
CE_compile_main( HWND hwndInput, HWND hwndError, LPCTSTR szPathname, LPTSTR pRcxPathname )
{
	RCX_Image	*image ;
	char		tmp[ 256 ] ;
	TCHAR		szRcxPathname[ MAX_PATH ], *ptr ;

	SetupCompilerOption() ;
	AttachWinToStream( CE_stdin, hwndInput ) ;
	AttachWinToStream( CE_stdout, hwndError ) ;
	AttachWinToStream( CE_stderr, hwndError ) ;
	fputs( "[Start compiling]\n", gErrorStream ) ;
	/* RpCs */
	image = Compile( NULL, g_fFlags ) ;
	if ( !image ) {
		int	errors = ErrorHandler::Get()->GetCount() ;
		if ( errors ) {
			fprintf( gErrorStream, "# %d error%s during compilation\n", errors, errors == 1 ? "" : "s" ) ;
		}
		return FALSE ;
	}
	/* o͗pRCXt@C쐬 */
	if ( *szPathname ) {
		_tcscpy( szRcxPathname, szPathname ) ;
		ptr = _tcsrchr( szRcxPathname, TEXT('.') ) ;
		if ( ptr && !_tcsicmp( ptr, kNQCFileExtension ) ) {
			_tcscpy( ptr, kRCXFileExtension ) ;
		} else {
			_tcscat( szRcxPathname, kRCXFileExtension ) ;
		}
	} else {
		_tcscpy( szRcxPathname, TEXT("\\cenqc") ) ;
		_tcscat( szRcxPathname, kRCXFileExtension ) ;
	}
	/* C[Wt@Cɏ */
#ifdef	UNICODE
	Wcstombs( tmp, szRcxPathname, 256 ) ;
#else	/* UNICODE */
	strcpy( tmp, szRcxPathname ) ;
#endif	/* UNICODE */
	fprintf( gErrorStream, "[Writing RCX code to '%s']\n", tmp ) ;
	if ( !image->Write( tmp ) ) {
		Fputs( "Ouput file error\n", gErrorStream ) ;
	}
	delete image ;
	Fputs( "[Compiling done]\n", gErrorStream ) ;
	/* o͗pRCXt@CԂ */
	if ( pRcxPathname ) {
		_tcscpy( pRcxPathname, szRcxPathname ) ;
	}
	return TRUE ;
}

BOOL
CE_download_main( LPCTSTR szPathname, HWND hwndError )
{
	RCX_Image	*image ;
	RCX_Result	result ;
	CHAR		tmp[ 256 ] ;

	SetupDownloadOption() ;
	AttachWinToStream( CE_stdout, hwndError ) ;
	AttachWinToStream( CE_stderr, hwndError ) ;
	fputs( "[Start downloading]\n", gErrorStream ) ;
	/* RCXt@Cǂ݂ */
	image = new RCX_Image() ;
#ifdef	UNICODE
	Wcstombs( tmp, szPathname, 256 ) ;
#else	/* UNICODE */
	strcpy( tmp, szPathname ) ;
#endif	/* UNICODE */
	result = image->Read( tmp ) ;
	if ( RCX_ERROR( result ) ) {
		PrintError( result, szPathname ) ;
		delete image ;
		return FALSE ;
	}
	/* RCXɓ] */
	RCX_Result r2 = Download( image ) ;
	if ( result == kRCX_OK ) {
		result = r2 ;
	}
	if ( result != kRCX_OK ) {
		PrintError( result, TEXT("") ) ;
	}
	fputs( "[Downloading done]\n", gErrorStream ) ;
	delete image ;
	return TRUE ;
}

BOOL
CE_compile_sub( HWND hwndInput, HWND hwndBuild )
{
	/* RpC */
	SetupCompilerOption() ;
	AttachWinToStream( CE_stdin, hwndInput ) ;
	AttachWinToStream( CE_stdout, hwndBuild ) ;
	AttachWinToStream( CE_stderr, hwndBuild ) ;
	fputs( "[Start compiling]\n", gErrorStream ) ;
	/* RpCs */
	g_image = Compile( NULL, g_fFlags ) ;
	if ( !g_image ) {
		int	errors = ErrorHandler::Get()->GetCount() ;
		if ( errors ) {
			fprintf( gErrorStream, "# %d error%s during compilation\n", errors, errors == 1 ? "" : "s" ) ;
		}
		return FALSE ;
	}
	fputs( "[Compiling done]\n", gErrorStream ) ;
	return TRUE ;
}

BOOL
CE_download_sub( HWND hwndDownload )
{
	/* _E[h */
	if ( !g_image ) {
		return FALSE ;
	}
	SetupDownloadOption() ;
	AttachWinToStream( CE_stdout, hwndDownload ) ;
	AttachWinToStream( CE_stderr, hwndDownload ) ;
	Fputs( "[Start downloading]\n", gErrorStream ) ;
	RCX_Result result = Download( g_image ) ;
	if ( result != kRCX_OK ) {
		PrintError( result, TEXT("") ) ;
	}
	Fputs( "[Downloading done]\n", gErrorStream ) ;
	delete g_image ;
	g_image = 0 ;
	return TRUE ;
}

BOOL
CE_change_program( int nNo, HWND hwndDownload )
{
	RCX_Cmd		cmd ;
	RCX_Result	result ;

	SetupDownloadOption() ;
	AttachWinToStream( CE_stdout, hwndDownload ) ;
	AttachWinToStream( CE_stderr, hwndDownload ) ;
	Fprintf( CE_stderr, "[Change program to %d]\n", nNo + 1 ) ;
	result = gLink.Send( cmd.Set( kRCX_SelectProgramOp, (UByte) nNo ) ) ;
	gLink.Close() ;
	PrintError( result, "" ) ;
	return TRUE ;
}

BOOL
CE_run_program( BOOL bRun, HWND hwndDownload )
{
	RCX_Cmd		cmd ;
	RCX_Result	result ;

	SetupDownloadOption() ;
	AttachWinToStream( CE_stdout, hwndDownload ) ;
	AttachWinToStream( CE_stderr, hwndDownload ) ;
	Fprintf( CE_stderr, "[%s program]\n", bRun ? "Run" : "Stop" ) ;
	if ( bRun ) {
		result = gLink.Send( cmd.Set( kRCX_StartTaskOp, 0 ) ) ;
	} else {
		result = gLink.Send( cmd.Set( kRCX_OutputModeOp, kRCX_OutputOff|0x07 ) ) ;
		if ( result == kRCX_OK ) {
			result = gLink.Send( cmd.Set( kRCX_StopAllOp ) ) ;
		}
	}
	gLink.Close() ;
	PrintError( result, "" ) ;
	return TRUE ;
}

static void
SetupCompilerOption()
{
	LPTSTR	token ;
	CHAR	szSjis[ 256 ] ;

	switch ( g_dwTargetMode ) {
	case 1:		gTargetType = kRCX_ScoutTarget ;	break ;
	case 2:		gTargetType = kRCX_CMTarget ;		break ;
	default:	gTargetType = kRCX_RCXTarget ;		break ;
	}
	g_fFlags = 0 ;
	g_fFlags = g_dwNoSysFile ? Compiler::kNoSysFile_Flag : 0 ;
	g_fFlags = g_dwCompatible ? Compiler::kCompat_Flag : 0 ;
	/* IncludePathݒ肷 */
	gMyCompiler.DeleteDir() ;
	token = _tcstok( g_szIncludePath, TEXT(";") ) ;
	while ( token != NULL ) {
#ifdef	UNICODE
		Wcstombs( szSjis, token, 256 ) ;
#else	/* UNICODE */
		strcpy( szSjis, token ) ;
#endif	/* UNICODE */
		gMyCompiler.AddDir( szSjis ) ;
		token = _tcstok( NULL, TEXT(";") ) ;
	}
}

static void
SetupDownloadOption()
{
	switch ( g_dwTargetMode ) {
	case 1:		gTargetType = kRCX_ScoutTarget ;	break ;
	case 2:		gTargetType = kRCX_CMTarget ;		break ;
	default:	gTargetType = kRCX_RCXTarget ;		break ;
	}
#ifdef	UNICODE
	Wcstombs( g_sjisSerialPort, g_szSerialPort, 256 ) ;
#else	/* UNICODE */
	strcpy( g_sjisSerialPort, g_szSerialPort ) ;
#endif	/* UNICODE */
	gLink.SetSerialPort( g_sjisSerialPort ) ;
	gLink.SetRxTimeout( g_dwRxTimeout ) ;
	gLink.SetVerbose( g_dwVerbose ? true : false ) ;
}
