/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#ifdef	WINCE
#include	"cetools.h"
#else	/* WINCE */
#include <stdio.h>
#include <string.h>
#endif	/* WINCE */
#include "RCX_Disasm.h"
#include "RCX_Constants.h"

static void SPrintOutputArgs(char *argText, const UByte *code);
static void SPrintValue(char *value, int type, short data);
static void SPrintTest(char *text, const UByte *code, UShort pc);

static int ComputeOffset(UByte b1, UByte b2=0);

#define LOOKUP(i,a)	(((unsigned)(i)<sizeof(a)/sizeof(char*)) ? a[i] : "?")
#define WORD(ptr)	((short)((((ptr)[1]) << 8) + ((ptr)[0])))

static const char *inputTypeNames[] = { "None", "Switch", "Temp", "Light", "Angle" };
static const char *inputModeNames[] = { "Raw", "Boolean", "Edge", "Pulse", "Percent", "Celcius", "Fahrenheit", "Angle" };
static const char *outputDirName[] = { "Rev", "Flip", "Fwd" };
static const char *outputModeName[] = { "Float", "Off", "On" };
static const char *relNames[] = { "<=", ">=", "!=", "==" };
static const char *typeNames[] = {
	"Var", "Timer", "Const", "%3", "Random", "TachoCount", "TachoSpeed", "MotorCurrent",
	"%8", "Input", "%10", "%11", "%12", "%13", "Watch", "Message",
	"AGC" };

RCX_Disasm gRCX_Disasm;


enum Format
{
	kFormat_Unknown=0,
	kFormat_InputType,
	kFormat_InputMode,
	kFormat_Output,
	kFormat_Raw8,
	kFormat_Raw16,
	kFormat_Value8,
	kFormat_Value16,
	kFormat_TestL,
	kFormat_JumpL,
	kFormat_CheckLoopL,
	kFormat_JumpS,
	kFormat_Tone,
	kFormat_Var,
	kFormat_UploadDatalog,
	kFormat_SetWatch,
	kFormat_Raw8Value16,
	kFormat_Null8Raw16,
	kFormat_VarRaw8,
	kFormat_DecCheckL
};

class Instruction
{
public:
	const char*	fName;
	UByte		fOpcode;
	UByte		fLength;
	Format		fFormat;
};


static Instruction sInitData[] = {
{ "out", kRCX_OutputModeOp, 2, kFormat_Output },
{ "pwr", kRCX_OutputPowerOp, 4, kFormat_Output },
{ "dir", kRCX_OutputDirOp, 2, kFormat_Output },
{ "InMode", kRCX_InputModeOp, 3, kFormat_InputMode },
{ "InType", kRCX_InputTypeOp, 3, kFormat_InputType },
{ "wait", kRCX_DelayOp, 4, kFormat_Value16 },
{ "chkl", kRCX_TestOp, 8, kFormat_TestL },
{ "jmpl", kRCX_JumpOp, 3, kFormat_JumpL},
{ "jmp", kRCX_SJumpOp, 2, kFormat_JumpS},
{ "playt", kRCX_PlayToneOp, 4, kFormat_Tone },
{ "plays", kRCX_PlaySoundOp, 2, kFormat_Raw8},
{ "Display", kRCX_DisplayOp, 4, kFormat_Value16},
{ "setv", kRCX_VarOp(kRCX_SetVar), 5, kFormat_Var},
{ "sumv", kRCX_VarOp(kRCX_AddVar), 5, kFormat_Var},
{ "subv", kRCX_VarOp(kRCX_SubVar), 5, kFormat_Var},
{ "divv", kRCX_VarOp(kRCX_DivVar), 5, kFormat_Var},
{ "mulv", kRCX_VarOp(kRCX_MulVar), 5, kFormat_Var},
{ "andv", kRCX_VarOp(kRCX_AndVar), 5, kFormat_Var},
{ "orv", kRCX_VarOp(kRCX_OrVar), 5, kFormat_Var},
{ "absv", kRCX_VarOp(kRCX_AbsVar), 5, kFormat_Var},
{ "sgnv", kRCX_VarOp(kRCX_SgnVar), 5, kFormat_Var},
{ "start", kRCX_StartTaskOp, 2, kFormat_Raw8},
{ "stop", kRCX_StopTaskOp, 2, kFormat_Raw8},
{ "StopAll", kRCX_StopAllOp, 1},
{ "tmrz", kRCX_ClearTimerOp, 2, kFormat_Raw8},
{ "calls", kRCX_GoSubOp, 2, kFormat_Raw8},
{ "SetLoop", kRCX_SetLoopOp, 3, kFormat_Value8},
{ "CheckLoop", kRCX_CheckLoopOp, 3 , kFormat_CheckLoopL },
{ "msgz", kRCX_ClearMsgOp, 1 },
{ "msg", kRCX_SendMsgOp, 3, kFormat_Value8 },
{ "ClearSensor", kRCX_ClearSensorOp, 2, kFormat_Raw8},
{ "SetLog", kRCX_SetDatalogOp, 3, kFormat_Raw16},
{ "Datalog", kRCX_DatalogOp, 3, kFormat_Value8},
{ "UploadLog", kRCX_UploadDatalogOp, 5, kFormat_UploadDatalog},
{ "Drive", kRCX_DriveOp, 2 },
{ "OnWait", kRCX_OnWaitOp, 3 },
{ "OnWaitDifferent", kRCX_OnWaitDifferentOp, 4 },
{ "ClearTacho", kRCX_ClearTachoOp, 2 },
{ "SetWatch", kRCX_SetWatchOp, 3, kFormat_SetWatch},
{ "txs", kRCX_IRModeOp, 2, kFormat_Raw8},

// Scout specific bytecodes
{ "cntd", 0xa7, 2, kFormat_Raw8 },
{ "cnti", 0x97, 2, kFormat_Raw8 },
{ "cnts", 0xd4, 5, kFormat_Raw8Value16 },
{ "cntz", 0xb7, 2, kFormat_Raw8 },
{ "decvjn", 0xf2, 3 },
{ "decvjnl", 0xf3, 4, kFormat_DecCheckL },
{ "event", 0x03, 4, kFormat_Null8Raw16 },
{ "gdir", kRCX_GOutputDirOp, 2, kFormat_Output},
{ "gout", kRCX_GOutputModeOp, 2, kFormat_Output},
{ "gpwr", kRCX_GOutputPowerOp, 4, kFormat_Output},
{ "light", 0x87, 2, kFormat_Raw8 },
{ "lsbt", 0xe3, 4, kFormat_Value16 },
{ "lscal", 0xc0, 1 },
{ "lsh", 0xd3, 4, kFormat_Value16 },
{ "lslt", 0xc3, 4, kFormat_Value16 },
{ "lsut", 0xb3, 4, kFormat_Value16 },
{ "monal", 0x73, 4 },
{ "monax", 0xa0, 1, },
{ "mone", 0xb4, 5 },
{ "monel", 0xb5, 6 },
{ "monex", 0xb0, 1 },
{ "playv", 0x02, 3, kFormat_VarRaw8 },
{ "pollm", 0x63, 4 },
{ "remote", 0xd2, 3, kFormat_Raw16 },
{ "rules", 0xd5, 6 },
{ "scout", 0x47, 2, kFormat_Raw8 },
{ "setfb", 0x83, 4, kFormat_Value16 },
{ "setp", 0xd7, 2, kFormat_Raw8 },
{ "sound", 0x57, 2, kFormat_Raw8 },
{ "tmrs", 0xc4, 5, kFormat_Raw8Value16 },
{ "vll", 0xe2, 3, kFormat_Value8 },
{ "rts", 0xf6, 1 }
};

static Instruction *sDispatch[256] = { 0};
static int sInited = 0;

void InitDispatch();

RCX_Disasm::RCX_Disasm()
{
	InitDispatch();
}


void RCX_Disasm::Print(RCX_Printer *dst, const UByte *code, int length)
{
	char text[256];
	char line[256];
	UShort pc = 0;
	RCX_Result result;
	
	while(length > 0)
	{
		result = SPrint1(text, code, length, pc);
		
		if (result < 1)
		{
			result = 1;
			sprintf(text, "?");
		}

		sprintf(line, "%03d %-32s ", pc, text);
		dst->Print(line);
		
		for(int i=0; i<result; i++)
		{
			sprintf(line, "%02x ", code[i]);
			dst->Print(line);
		}
		sprintf(line, "\n");
		dst->Print(line);
		
		pc += result;
		code += result;
		length -= result;
	}
}	


RCX_Result RCX_Disasm::SPrint1(char *text, const UByte *code, int length, UShort pc)
{
	int iLength;
	Instruction *inst;
	char argText[256];
	UByte op;
	
	InitDispatch();
	
	if (length < 1) return -1;
	
	op = code[0];
	inst = sDispatch[op];
	if (!inst) return -1;
	
	iLength = inst->fLength;
	if (length < iLength) return -1;

	// process args
	switch(inst->fFormat)
	{
		case kFormat_InputType:
			sprintf(argText,"%d, %s", code[1],  LOOKUP(code[2], inputTypeNames));
			break;
		case kFormat_InputMode:
			sprintf(argText,"%d, %s", code[1],  LOOKUP((code[2] >> 5) & 7, inputModeNames));
			break;
		case kFormat_Output:
			SPrintOutputArgs(argText, code);
			break;
		case kFormat_Value16:
			// 16 bit value
			SPrintValue(argText, code[1], WORD(code+2));
			break;
		case kFormat_TestL:
			SPrintTest(argText, code, pc);
			break;
		case kFormat_JumpL:
			sprintf(argText,"%d", pc + 1 + ComputeOffset(code[1], code[2])); 
			break;
		case kFormat_CheckLoopL:
			sprintf(argText,"%d", pc + 1 + (short)WORD(code+1));
			break;
		case kFormat_JumpS:
			sprintf(argText,"%d", pc + 1 + ComputeOffset(code[1])); 
			break;
		case kFormat_Tone:
			sprintf(argText,"%d, %d", code[1] + (code[2] << 8), code[3]);
			break;
		case kFormat_Raw8:
			// 8 bit raw
			sprintf(argText, "%d", code[1]);
			break;
		case kFormat_Var:
			sprintf(argText, "var[%d], ", code[1]);
			SPrintValue(argText + strlen(argText), code[2], WORD(code+3));
			break;
		case kFormat_Value8:
			// 8 bit value
			SPrintValue(argText, code[1], code[2]);
			break;
		case kFormat_Raw16:
			// 16 bit raw
			sprintf(argText, "%d", WORD(code+1));
			break;
		case kFormat_UploadDatalog:
			sprintf(argText, "%d, %d", WORD(code+1), WORD(code+3));
			break;
		case kFormat_SetWatch:
			sprintf(argText, "%02d:%02d", code[1], code[2]);
			break;
		case kFormat_Raw8Value16:
			sprintf(argText, "%d, ", code[1]);
			SPrintValue(argText + strlen(argText), code[2], WORD(code+3));
			break;
		case kFormat_Null8Raw16:
			sprintf(argText, "%d", WORD(code+2));
			break;
		case kFormat_VarRaw8:
			sprintf(argText, "var[%d], %d", code[1], code[2]);
			break;
		case kFormat_DecCheckL:
			sprintf(argText,"var[%d], %d", code[1], pc + 2 + (short)WORD(code+2)); 
			break;
		default:
			argText[0] = 0;
			break;
	}
	
	sprintf(text,"%-10s %s", inst->fName, argText);

	return iLength;
}


void InitDispatch()
{
	unsigned i;
	
	if (sInited) return;
	
	for(i=0; i < sizeof(sInitData) / sizeof(Instruction); i++)
	{
		sDispatch[sInitData[i].fOpcode ^ 0x8] = &sInitData[i];
		sDispatch[sInitData[i].fOpcode] = &sInitData[i];
	}
	
	sInited = 1;
}



void SPrintOutputArgs(char *argText, const UByte *code)
{
	char *ptr = argText;
	
	// list outputs
	for(int i=0; i<3; i++)
		if (code[1] & (1 << i))
			*ptr++ = (char)('A' + i);
	*ptr++ = ',';
	*ptr++ = ' ';
	
	switch(code[0])
	{
		case kRCX_OutputDirOp:
		case kRCX_GOutputDirOp:
			strcpy(ptr, LOOKUP((code[1] >> 6) & 3, outputDirName));
			break;
		case kRCX_OutputModeOp:
		case kRCX_GOutputModeOp:
			strcpy(ptr, LOOKUP((code[1] >> 6) & 3, outputModeName));
			break;
		case kRCX_OutputPowerOp:
		case kRCX_GOutputPowerOp:
			SPrintValue(ptr, code[2], code[3]);
			break;
		default:
			*ptr = 0;
			break;
	}
}



void SPrintValue(char *text, int type, short data)
{
	switch(type)
	{
		case 0:
			sprintf(text, "var[%d]", data);
			break;
		case 2:
			sprintf(text,"%d", data);
			break;
		default:
			sprintf(text, "%s(%d)", LOOKUP(type, typeNames), data);
			break;
	}
}


void SPrintTest(char *text, const UByte *code, UShort pc)
{
	char v1Text[16];
	char v2Text[16];
	short offset;
	
	offset = (short)((code[7] << 8) + code[6]);
		
	SPrintValue(v1Text, (code[1] & 0xf), (short)((code[4]<<8) + code[3]));
	SPrintValue(v2Text, (code[2] & 0xf), code[5]); 
	sprintf(text, "%s %s %s, %d", v1Text, LOOKUP((code[1]>>6) & 3, relNames),
		v2Text, pc + 6 + offset);
}


int ComputeOffset(UByte b1, UByte b2)
{
	int x = (b1 & 0x7f) + (b2 << 7);
	
	if (b1 & 0x80) x = -x;
	
	return x;
}
