%{
/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */
%}

%{
// prevent redefinition of YYSTYPE in parser.h
#define __PARSE_TAB_H

// these classes must be defined prior to YYSTYPE
#include "Fragment.h"
#include "BlockStmt.h"
#include "Condition.h"
#include "Symbol.h"
#include "Expr.h"
#include "Function.h"
#include "CallStmt.h"
#include "AsmStmt.h"
#include "LabeledStmt.h"

class Clause;

%}

%union {
	int			fInt;
	Fragment*	fFragment;
	Stmt*		fStmt;
	BlockStmt*	fBlock;
	Symbol*		fSymbol;
	Condition*	fCondition;	
	char*		fString;
	Expr*		fExpr;
	Clause*		fClause;
	Function*	fFunction;
	CallStmt*	fCall;
	AsmStmt*	fAsmStmt;
	Field*		fField;
	LabeledStmt*	fLabeledStmt;
}


%{
#include <stdlib.h>
#include "IfStmt.h"
#include "WhileStmt.h"
#include "AsmStmt.h"
#include "DoStmt.h"
#include "RepeatStmt.h"
#include "AssignStmt.h"
#include "TaskStmt.h"
#include "JumpStmt.h"
#include "ReturnStmt.h"
#include "SwitchStmt.h"
#include "ExprStmt.h"
#include "RCX_Constants.h"
#include "PreProc.h"
#include "parser.h"
#include "Program.h"
#include "parse_util.h"
#include "Clause.h"
#include "Program.h"
#include "IncDecExpr.h"
#include "AtomExpr.h"
#include "TypeExpr.h"
#include "Error.h"

#define yylex()		(gPreProc->Get(yylval))

static LexLocation sSavedLoc;

%}


%left OR
%left AND
%left '|'
%left '^'
%left '&'
%left LEFT RIGHT
%left '-' '+'
%left '*' '/' '%'
%right UMINUS '~'
%left INCDEC

%nonassoc ABS SIGN TYPE
%nonassoc LOWER_THAN_ELSE
%nonassoc ELSE

%nonassoc LOWER_THAN_EXPR_SHIFT
%nonassoc ')'

%token <fSymbol> ID
%token <fInt> NUMBER ASSIGN TASKOP JUMP TASK SUB INCDEC
%token <fString> STRING
%token REL_GE REL_LE REL_EQ REL_NE

%token PP_DEFINE PP_INCLUDE NL WS PP_ARG PP_ERROR PP_IFDEF PP_IF PP_ELSE PP_ELIF PP_ENDIF PP_UNDEF PP_PRAGMA PP_GLOM
%token IF WHILE DO REPEAT RETURN SWITCH CASE DEFAULT
%token ASM
%token INT INLINE T_VOID T_CONST SENSOR
%token CTRUE CFALSE

%type <fStmt> stmt var_decl
%type <fBlock> stmt_list block var_list
%type <fInt> S relation arg_type
%type <fCondition> condition
%type <fSymbol> id
%type <fExpr> expr
%type <fClause> clause
%type <fFunction> args arg_list
%type <fCall> params param_list
%type <fAsmStmt> asm_list
%type <fField> asm_item
%type <fLabeledStmt> label


%{
static LabeledStmt* MakeCaseStmt(const Expr *e);
static void DefineArg(Function *f, const Symbol *name, int type);
static Function* BeginFunction(Function *f, const Symbol *name);
static void EndFunction(Function *f, Stmt *body);
static ConstField* MakeConstField(Expr *e);
static Expr *MakeVarExpr(const Symbol *name);
static void yyerror(const char *msg);

%}


%%

S : unit_list	{ $$ = 0; }
	;


unit_list : unit_list unit
	|
	;

unit :	INT var_list ';'		{ gProgram->AddInitStmt($2); }
	|	TASK ID  opt_arg block	{ (void)new Fragment(kRCX_TaskFragment, $2, $4); gProgram->ExclusiveVar(); }
	|	SUB { gProgram->SetContext(0); } ID opt_arg block
				{ (void)new Fragment(kRCX_SubFragment, $3, $5); gProgram->DefaultContext(); }
				
	|	INLINE ID  { Error(kDep_Inline, $2->GetKey()).RaiseLex();
			$<fFunction>$ = BeginFunction(new Function(), $2); } '{' stmt_list '}' { EndFunction($<fFunction>3, $5); }
	|	T_VOID ID '(' args ')' { BeginFunction($4, $2); } '{' stmt_list '}'  { EndFunction($4, $8); }
	;


opt_arg : '(' ')'
	|				{ Error(kDep_NoArgList).RaiseLex(); }
	;

args : arg_list
	| T_VOID	{ $$ = new Function(); }
	|			{ $$ = new Function(); }
	;

arg_list : arg_list ',' arg_type ID	{ $$ = $1; 	DefineArg($$, $4, $3); }
	| arg_type ID					{ $$ = new Function(); DefineArg($$,$2,$1); }
	;

arg_type : INT		{ $$ = Function::kIntegerArg; }
	| T_CONST INT 	{ $$ = Function::kConstantArg; }
	| INT '&'		{ $$ = Function::kReferenceArg; }
	| T_CONST INT '&'	{ $$ = Function::kConstRefArg; }
	| SENSOR		{ $$ = Function::kSensorArg; }
	;

var_list : var_list ',' var_decl	{ $1->Add($3); $$ = $1; }
	| var_decl						{ $$ = new BlockStmt(); $$->Add($1); }
	;

var_decl : ID		{ gProgram->CreateVar($1); $$ = 0; }
	|	ID '=' clause { $$ = new AssignStmt(gProgram->CreateVar($1), kRCX_SetVar ,$3); }
	;

block : '{' { gProgram->PushScope(); } stmt_list { gProgram->PopScope(); } '}'		{ $$ = $3; }
	;

stmt_list : stmt_list  stmt 	{ $1->Add($2); $$ = $1; }
	|							{ $$ = new BlockStmt(); }
	;

stmt :	';'										{ $$ = new BlockStmt(); }
	|	block		{ $$ = $1; }
	|	ASM '{' asm_list '}' ';'				{ $$ = $3; }
	|	WHILE '(' condition ')' stmt			{ $$ = new WhileStmt($3, $5); }
	|	DO stmt WHILE '(' condition ')' ';'			{ $$ = new DoStmt($5, $2); }
	|	REPEAT '(' clause ')' stmt				{ $$ = new RepeatStmt($3, $5); }
	|	SWITCH '(' clause ')' stmt				{ $$ = new SwitchStmt($3, $5); }
	|	label stmt								{ $$ = $1; $1->SetStmt($2); }
	|	loc JUMP ';'							{ $$ = new JumpStmt($2, sSavedLoc); }
	|	IF	'(' condition ')' stmt	%prec LOWER_THAN_ELSE	{ $$ = new IfStmt($3, $5); }
	|	IF '(' condition ')' stmt ELSE stmt		{ $$ = new IfStmt($3, $5, $7); }
	|	expr ASSIGN clause ';'				{ $$ = new AssignStmt(GetLValue($1), (RCX_VarCode)$2, $3); }
	|	expr '=' clause ';'					{ $$ = new AssignStmt(GetLValue($1), kRCX_SetVar ,$3); }
	|	TASKOP id ';'  						{ $$ = new TaskStmt((UByte)$1, $2, sSavedLoc); }
	|	id '(' params ')' ';'		{ $$ = $3; $3->SetName($1); $3->SetLocation(sSavedLoc); }
	|	RETURN ';'								{ $$ = new ReturnStmt(); }
	|	error ';'	{  yyerrok; $$ = new BlockStmt(); }
	|	clause ';'	{ $$ = new ExprStmt($1); }
	|	INT	var_list ';'	{ $$ = $2; }
	;

label : loc CASE expr ':'	{ $$ = MakeCaseStmt($3); }
	  | loc DEFAULT ':'		{ $$ = new LabeledStmt(LabeledStmt::kDefaultValue, sSavedLoc); }
	  ;

params : param_list
	|				{ $$ = new CallStmt(); }
	;

param_list : param_list ',' clause	{ $$ = $1; $$->AddParam($3); }
	| clause						{ $$ = new CallStmt(); $$->AddParam($1); }
	;

clause : expr loc					{ $$ = new Clause($1, sSavedLoc); }
	;

expr : NUMBER			{ $$ = new AtomExpr(kRCX_ConstantType, $1); }
	|	expr '+' expr	{ $$ = MakeBinaryExpr($1, '+', $3); }
	|	expr '-' expr	{ $$ = MakeBinaryExpr($1, '-', $3); }
	|	expr '*' expr	{ $$ = MakeBinaryExpr($1, '*', $3); }
	|	expr '/' expr	{ $$ = MakeBinaryExpr($1, '/', $3); }
	|	expr '&' expr	{ $$ = MakeBinaryExpr($1, '&', $3); }
	|	expr '|' expr	{ $$ = MakeBinaryExpr($1, '|', $3); }
	
	|	expr '%' expr	{ $$ = MakeBinaryExpr($1, '%', $3); }
	|	expr LEFT expr	{ $$ = MakeBinaryExpr($1, LEFT, $3); }
	|	expr RIGHT expr	{ $$ = MakeBinaryExpr($1, RIGHT, $3); }
	|	expr '^' expr	{ $$ = MakeBinaryExpr($1, '^', $3); }

	|	'-'	expr %prec UMINUS	{ $$ = MakeBinaryExpr(new AtomExpr(kRCX_ConstantType, 0), '-', $2); }
	|	'~' expr %prec UMINUS	{ $$ = MakeUnaryExpr('~', $2); }
	
	|	ABS '(' expr ')' 	{ $$ = MakeUnaryExpr(ABS, $3); }
	|	SIGN '(' expr ')'	{ $$ = MakeUnaryExpr(SIGN, $3); }
	
	| '(' expr ')'			{ $$ = $2; }
	| id			{ $$ = MakeVarExpr($1); }
	| '@' expr %prec UMINUS			{ $$ = MakeValueExpr($2); }
	
	| expr INCDEC				{ $$ = new IncDecExpr(GetLValue($1), $2, false); }
	| INCDEC expr %prec UMINUS	{ $$ = new IncDecExpr(GetLValue($2), $1, true); }
	
	| TYPE '(' expr ')'			{ $$ = new TypeExpr($3); }
	;
	
loc :  %prec LOWER_THAN_EXPR_SHIFT		{ LexCurrentLocation(sSavedLoc); }
	;

id : { LexCurrentLocation(sSavedLoc); } ID { $$ = $2; }
	;

asm_list : asm_list ',' asm_item	{ $$ = $1; $1->Add($3); }
	| asm_item	{ $$ = new AsmStmt(); $$->Add($1); }
	;


asm_item : expr		{ $$ = MakeConstField($1); }
	| '&' clause	{ $$ = new EAField($2, 0xffff); }
	| '&' clause ':' expr	{ $$ = new EAField($2, GetConstantValue($4)); }
	;

condition : clause relation clause	{ $$ = new TestCond($1, $2, $3); }
	| '!' condition	%prec UMINUS	{ $$ = $2; $2->Invert(); }
	| condition AND condition		{ $$ = new CompCond($1, kCompAnd, $3); }
	| condition OR condition		{ $$ = new CompCond($1, kCompOr, $3); }
	| '(' condition ')'				{ $$ = $2; }
	| CTRUE							{ $$ = new ConstCond(true); }
	| CFALSE						{ $$ = new ConstCond(false); }
	| clause						{ $$ = new TestCond($1, kNotEqualTo, new Clause(new AtomExpr(kRCX_ConstantType, 0), sSavedLoc));  }
	;

relation :	'<'		{ $$ = kLessThan; }
	|		'>'		{ $$ = kGreaterThan; }
	|		REL_EQ	{ $$ = kEqualTo; }
	|		REL_NE	{ $$ = kNotEqualTo; }
	|		REL_GE	{ $$ = kGreaterOrEqual; }
	|		REL_LE	{ $$ = kLessOrEqual; }
	;
	
%%


void yyerror(const char *msg)
{
	Error(kErr_Parser, msg).RaiseLex();
}


LabeledStmt* MakeCaseStmt(const Expr *e)
{
	int v = GetConstantValue(e);
	
	if (v < -32768 || v > 32767)
		Error(kErr_CaseRange).RaiseLex();

	return new LabeledStmt(v, sSavedLoc);
}


void DefineArg(Function *f, const Symbol *name, int type)
{
	if (!f->AddArg(name, (Function::ArgType)type))
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
}


Function* BeginFunction(Function *f, const Symbol *name)
{
	f->SetName(name);
	gProgram->AddFunction(f);
	gProgram->SetContext(f);
	f->PrepareScope(gProgram->PushScope());
	return f;
}


void EndFunction(Function *f, Stmt *body)
{
	f->SetBody(body);
	gProgram->PopScope();
	gProgram->DefaultContext();
}


ConstField *MakeConstField(Expr *e)
{
	ConstField *c = new ConstField(e);
	
	if (!e->PromiseConstant())
	{
		Error(kErr_ConstantNeeded).RaiseLex();
	}
	return c;
}

Expr *MakeVarExpr(const Symbol *name)
{
	int var = gProgram->GetVar(name);
	if (var == kIllegalVar)
	{
		Error(kErr_UndefinedVar, name->GetKey()).RaiseLex();
	}
	
	return new AtomExpr(kRCX_VariableType, var);
}
