/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */


#ifdef	WINCE
#include	"cetools.h"
#else	/* WINCE */
#include <stdio.h>
#include <string.h>
#endif	/* WINCE */
#include "Buffer.h"

#define kDefaultBufferSize 10

Buffer::Buffer() :
	fName(0),
	fData(0),
	fLength(0)
{
}


Buffer::~Buffer()
{
	delete [] fName;
	delete [] fData;
}


bool Buffer::Create(const char *name, const char *pathname)
{
	FILE *fp = fopen(pathname, "rb");
	if (!fp) return false;
	
	Create(name, fp);
	fclose(fp);

	return true;	
}


void Buffer::Create(const char *name, FILE *fp)
{
	// figure out length
	if (fseek(fp,0,SEEK_END)==0)
	{
		// if stream supports seeking, then we can determine total size,
		// allcoate enough storage, and read the entire thing at once
#ifdef	WINCE
		fLength = Fsize( fp ) ;
#else	/* WINCE */
		fLength = ftell(fp);
		fseek(fp, 0, SEEK_SET);
#endif	/* WINCE */

		// allow room for terminating NL
		int allocate = fLength + 1;
		fData = new char[allocate];
		
		fread(fData, 1, fLength, fp);
	}
	else
	{
		// This code is for io streams (such as redirected stdin under MPW)
		// which don't support seeking.  The data from the stream is read into
		// a dynamically growing array.  This is a lot less efficient than
		// the previous case which knew the total lenght beforehand.
		int max = kDefaultBufferSize;
		fData = new char[max];
		fLength = 0;
		
		int c;
		while((c=getc(fp)) != EOF)
		{
			fData[fLength++] = (char)c;

			if (fLength == max)
			{
				// grow buffer
				max *= 2;
				char *newBuf = new char[max];
				memcpy(newBuf, fData, fLength);
				delete [] fData;
				fData = newBuf;
			}
		}
	}
	FinishCreate(name);	
}


void Buffer::Create(const char *name, const char *data, int length)
{	
	// allcoate memory (allow for trailing NL)
	fData = new char[length + 1];
	fLength = length;
	memcpy(fData, data, length);

	FinishCreate(name);	
}


void Buffer::FinishCreate(const char *name)
{
	fName = new char[strlen(name) + 1];
	strcpy(fName, name);
	
	fData[fLength++] = '\n';	
	ConvertLineEndings();
}


const char *Buffer::GetLine(int line) const
{
	if (line < 1) return 0;
	
	const char *ptr = fData;
	const char *end = fData + fLength;

	--line;
	while(line)
	{
		if (ptr == end) return 0;
		if (*ptr++ == '\n')
			--line;
	}	

	return ptr;
}


#define CR	0x0d
#define LF	0x0a

void Buffer::ConvertLineEndings()
{
	const char *src, *end;
	char *dst;
	
	src = fData;
	end = src + fLength;
	dst = fData;
	
	while(src < end)
	{
		char c = *src++;
		
		switch(c)
		{
			case CR:
				*dst++ = '\n';
				// skip second part of CR-LF pair
				if (src < end && *src==LF)
					src++;
				break;			
			case LF:
				*dst++ = '\n';
				break;
			default:
				*dst++ = c;
				break;
		}		
	}
	
	fLength = dst - fData;
}


