/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include "PArray.h"


P_Array::P_Array(int itemSize)
{
	fItemSize = itemSize;
	fGrowStep = kPArray_DoubleGrowStep;
	fLength = 0;
	fAllocated = 0;
	fContents = 0;
}


P_Array::~P_Array()
{
	SetLength(0);
	delete [] fContents;
}


void P_Array::SetLength(int count)
{
	if (count > fLength)
	{
		// grow the array

		if (count > fAllocated)
		{
			// how many items should be allocated?
			int n;
			
			if (fGrowStep)
				n = fAllocated + fGrowStep;
			else
				n = fAllocated * 2;

			if (n < count) n = count;
			
			Allocate(n);
		}
		
		Construct(fLength, count-fLength);
		fLength = count;
	}
	else if (count < fLength)
	{
		// shrink the array
		Destruct(count, fLength-count);
		fLength = count;
	}
	
}





void P_Array::Remove(int index, int count)
{
	if (index < 0 || index >= fLength) return;

	// can't remove more than we have
	if (index + count > fLength)
		count = fLength - index;
	
	char *ptr = fContents+index*fItemSize;
	int nCopy = fLength - index - count;
	
	memmove(ptr, ptr+fItemSize*count, (size_t)nCopy * fItemSize);

	Destruct(fLength-count, count);
	fLength -= count;
}


void P_Array::InsertSpace(int index, int count)
{
	if (index < 0 || index >= fLength) return;

	int nCopy;
	char *ptr = fContents+index*fItemSize;

	nCopy = fLength - index;
	Allocate(fLength+count);
	
	memmove(ptr+fItemSize * count, ptr, (size_t)nCopy * fItemSize);

	Construct(index, count);
	fLength += count;
}


void P_Array::Allocate(int n)
{
	char *ptr = 0;
	
	// truncate length if necessary
	if (fLength > n) fLength = n;
	
	// allocate and preserve data
	ptr = new char[n * fItemSize];
	
	if (fLength)
		memcpy(ptr, fContents, (size_t)fLength * fItemSize);
	
	// use new contents
	delete [] fContents;
	fContents = ptr;
	fAllocated = n;
}


void P_Array::Construct(int index, int count)
{
	int i;
	char *ptr;
	
	ptr = fContents + (fItemSize * index);
	for(i=0; i<count; i++)
	{
		Construct(ptr);
		ptr += fItemSize;
	}
}


void P_Array::Destruct(int index, int count)
{
	int i;
	char *ptr;

	ptr = fContents + (fItemSize * index);
	for(i=0; i<count; i++)
	{
		Destruct(ptr);
		ptr += fItemSize;
	}
}



