/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "RepeatStmt.h"
#include "Bytecode.h"
#include "CheckState.h"
#include "RCX_Cmd.h"
#include "RCX_Target.h"
#include "Clause.h"
#include "Error.h"
#include "RCX_Constants.h"

#define REPEAT_MASK	(TYPEMASK(kRCX_VariableType) + \
					TYPEMASK(kRCX_ConstantType) + \
					TYPEMASK(kRCX_RandomType))


bool RepeatStmt::sRCXLoopInUse = false;

RepeatStmt::RepeatStmt(Clause *c, Stmt *s) :
	ChainStmt(s)
{
	fCount = c;
}


RepeatStmt::~RepeatStmt()
{
	delete fCount;
}


bool RepeatStmt::Check(CheckState &state)
{
	bool ok = true;
	
	if (!fBody->Check(state)) ok = false;
	
	const Expr *e = fCount->GetExpr();

	if (fBody->IsNullable() && e->PromiseConstant())
		fNullable = true;
	else
	{
		int n;
		if (e->Evaluate(n) && (n==0))
			fNullable = true;
/*
			else if (n < 0 || n > 255)
			{
				Error(kErr_BadRepeatCount).Raise(fCount->GetLoc());
				ok = false;
			}
*/
	}

	return ok;
}


void RepeatStmt::EmitActual(Bytecode &b)
{
	if (b.Target()->fType == kRCX_ScoutTarget)
		EmitScout(b);
	else
	{
		// see if loop is candidate for loop counter
		const Expr *e = fCount->GetExpr();
		int n;
		
		if (e->Evaluate(n) &&
			n >= 0 &&
			n < 256 &&
			!sRCXLoopInUse)
		{
			// mark the loop counter in use, then emit code
			sRCXLoopInUse = true;
			EmitRCXLoop(b);
			
			// loop counter is now available for future use
			sRCXLoopInUse = false;
		}
		else
		{
			// use a temp variable rather than the loop counter
			EmitRCXVar(b);
		}	
	}
}


/*
 * EmitRCXLoop(Bytecode &b) - emit code using the RCX loop counter.
 *   Since there is only one loop counter per task/sub, this cannot
 *   be used for nested repeats
 */

void RepeatStmt::EmitRCXLoop(Bytecode &b)
{
	/*
			set loop count
		cPos:
			check loop
			body
			jump -> cPos
	*/

	RCX_Cmd cmd;
	int cPos;
	RCX_Value ea;
	
	b.PushLoopContext();
	
	ea = fCount->EmitConstrained(b, REPEAT_MASK);
	cmd.MakeSetLoop(ea);
	b.Add(cmd);
	b.ReleaseTempEA(ea);
		
	cPos = b.GetLength();
	
	cmd.MakeCheckLoop(0);
	b.Add(cmd);
	b.AddFixup(kPatch_Normal, b.GetLength() - 2, b.GetBreakLabel());

	fBody->Emit(b);
	b.AddJump(b.GetContinueLabel());
	
	b.PopLoopContext(cPos, b.GetLength());
}


/*
 * EmitRCXVar(Bytecode &b) - emit using a temp variable and decrementing
 * 	it manually.
 */

void RepeatStmt::EmitRCXVar(Bytecode &b)
{
	/*
			setvar to loop count
		cPos:
			dec var
			check var
			body
			jump -> cPos
	*/

	RCX_Cmd cmd;
	int cPos;
	RCX_Value ea;
	
	b.PushLoopContext();
	
	// set the loop variable
	ea = fCount->EmitConstrained(b, TYPEMASK(kRCX_VariableType));
	
	// emit the decrement/check
	cPos = b.GetLength();
	b.AddTest(RCX_VALUE(kRCX_ConstantType, 0), kRCX_GreaterOrEqual, ea, b.GetBreakLabel());
	cmd.MakeVar(kRCX_SubVar, RCX_VALUE_DATA(ea), RCX_VALUE(kRCX_ConstantType, 1));
	b.Add(cmd);

	// body and jump back to check
	fBody->Emit(b);
	b.AddJump(b.GetContinueLabel());

	// done with temp and loop context
	b.ReleaseTempEA(ea);
	b.PopLoopContext(cPos, b.GetLength());
}


/*
 * EmitScout(Bytecode &b) - emit using the scout's decvjnl instruction
 */
 
void RepeatStmt::EmitScout(Bytecode &b)
{
	/*
			setvar to loop count
		cPos:
			check var
			body
			jump -> cPos
	*/

	RCX_Cmd cmd;
	int cPos;
	RCX_Value ea;
	
	b.PushLoopContext();
	
	ea = fCount->EmitConstrained(b, TYPEMASK(kRCX_VariableType));
		
	cPos = b.GetLength();
	
	cmd.Set(0xf3, RCX_VALUE_DATA(ea), 0, 0);
	b.Add(cmd);

	b.AddFixup(kPatch_Normal, b.GetLength() - 2, b.GetBreakLabel());

	fBody->Emit(b);
	b.AddJump(b.GetContinueLabel());
	
	b.ReleaseTempEA(ea);
	
	b.PopLoopContext(cPos, b.GetLength());
}


Stmt* RepeatStmt::Clone(Mapping *b) const
{
	return new RepeatStmt(fCount->Clone(b), fBody->Clone(b));
}
