/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#ifdef	WINCE
#include	"cetools.h"
#endif	/* WINCE */
#ifdef	_WIN32
#include "Fragment.h"
#include "RCX_Image.h"
#include "Bytecode.h"
#include "parser.h"
#include "Symbol.h"
#include "BlockStmt.h"
#include "InlineStmt.h"
#include "VarPool.h"
#include "Function.h"
#include "Error.h"
#include "RCX_Target.h"
#include "Program.h"
#else	/* _WIN32 */
#include "Program.h"
#include "Fragment.h"
#include "RCX_Image.h"
#include "Bytecode.h"
#include "parser.h"
#include "Symbol.h"
#include "BlockStmt.h"
#include "InlineStmt.h"
#include "VarPool.h"
#include "Function.h"
#include "Error.h"
#include "RCX_Target.h"
#endif	/* _WIN32 */

static void EncodeFragment(RCX_Image *image, Fragment *f, VarPool &varPool,
	bool temps, int index, const RCX_Target *target);


Program *gProgram = new Program();

Program::Program() :
	fContext(this)
{
	fCounts[0] = 1;	// leave room for main
	fCounts[1] = 0;
	fMainAdded = false;
	fVarCount = fVarNext = 0;
	fScopes.InsertHead(new Scope(this));
	fInitName = Symbol::Get("_init");
	fInitStmts = new BlockStmt();
	fVarNames = 0;
}


Program::~Program()
{
#ifdef	_WIN32
	Fragment	*f ;
	Scope		*s ;
	Function	*func ;

	while(f=fTasks.RemoveHead())
		delete f;

	while(f=fSubs.RemoveHead())
		delete f;
	
	while(s=fScopes.RemoveHead())
		delete s;

	while(func = fFunctions.RemoveHead())
		delete func;
#else	/* _WIN32 */
	while(Fragment *f=fTasks.RemoveHead())
		delete f;

	while(Fragment *f=fSubs.RemoveHead())
		delete f;
	
	while(Scope *s=fScopes.RemoveHead())
		delete s;

	while(Function *func = fFunctions.RemoveHead())
		delete func;
#endif	/* _WIN32 */
	
	delete [] fVarNames;
	
	delete fInitStmts;
}


int Program::AddMainTask(Fragment *f)
{
	CheckName(f->GetName());
	fMainAdded = true;
	fTasks.InsertHead(f);
	return 0;
}


int Program::AddTask(Fragment *f)
{
	CheckName(f->GetName());
	fTasks.InsertTail(f);
	return fCounts[kRCX_TaskFragment]++;
}


int Program::AddSubroutine(Fragment *f)
{
	CheckName(f->GetName());
	fSubs.InsertTail(f);
	return fCounts[kRCX_SubFragment]++;
}


void Program::AddFunction(Function *f)
{
	CheckName(f->GetName());
	fFunctions.InsertTail(f);
}


void Program::AddInitStmt(Stmt *s)
{
	fInitStmts->Add(s);
}


void Program::CheckName(const Symbol *name)
{
	if (Defined(name))
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
}


int Program::CreateVar(Symbol *name)
{
	if (!fContext)
	{
		Error(kErr_NoVarDecl).RaiseLex();
		return kIllegalVar;
	}
	
	Scope *s = fScopes.GetHead();
	
	if (s->Contains(name))
	{
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
		return kIllegalVar;
	}

	int var = fContext->AllocateVar(name);
	s->Define(name, var);

	return var;
}


int Program::AllocateVar(const Symbol *name)
{
	if (fVarNext == fTarget->fMaxGlobalVars)
	{
		Error(kErr_MaxVars, fTarget->fMaxGlobalVars).RaiseLex();
		return kIllegalVar;
	}

	int var = fVarNext++;
	if (fVarNext > fVarCount)
		fVarCount = fVarNext;
	
	fVarNames[var] = name;
	
	return var;
}


int Program::GetMark() const
{
	return fVarNext;
}

void Program::SetMark(int mark)
{
	fVarNext = mark;
}


int Program::GetVar(const Symbol *name)
{
	int var = fScopes.GetHead()->Lookup(name);
	if (var == kIllegalVar)
		Error(kErr_Undeclared, name->GetKey());
	
	return var;
}



void Program::SetTarget(const RCX_Target *target)
{
	fTarget = target;

	delete [] fVarNames;
	fVarNames = new const Symbol*[target->fMaxGlobalVars];
}


RCX_Image*	Program::CreateImage()
{
	RCX_Image *image;
	int index;	
	VarPool varPool(fTarget->fMaxGlobalVars);
	
	for(int i=0; i<fVarCount; i++)
		varPool.Reserve(i);

	if (!Check(fTarget, varPool)) return nil;
		
	image = new RCX_Image();
	
	image->SetFragmentCount(fCounts[0] + fCounts[1]);
	index = 0;

	// emit tasks
	for(Fragment *task=fTasks.GetHead(); task; task=task->GetNext())
	{
		EncodeFragment(image, task, varPool, true, index++, fTarget);

		// emit subs that fall within the task's execution space for temp vars
		for(Fragment *sub=fSubs.GetHead(); sub; sub=sub->GetNext())
		{
			if (sub->GetTaskID() == task->GetTaskID())
				EncodeFragment(image, sub, varPool, true, index++, fTarget);
		}

		varPool.ReserveUsed();
	}

	// emit other subs (multi or no task ID)
	for(Fragment *sub=fSubs.GetHead(); sub; sub=sub->GetNext())
	{
		if (sub->GetTaskID() == Fragment::kNoTaskID || sub->GetTaskID() == Fragment::kMultiTaskID)
			EncodeFragment(image, sub, varPool, false, index++, fTarget);
	}

	// emit global variable names
#ifdef	_WIN32
	for(i=0; i<fVarCount; i++)
#else	/* _WIN32 */
	for(int i=0; i<fVarCount; i++)
#endif	/* _WIN32 */
		image->SetVariable(i, fVarNames[i]->GetKey());

	// emit temp names
#ifdef	_WIN32
	for(i=fVarCount; i<varPool.GetMaxVars(); i++)
#else	/* _WIN32 */
	for(int i=fVarCount; i<varPool.GetMaxVars(); i++)
#endif	/* _WIN32 */
	{
		if (!varPool.IsFree(i))
		{
			char varName[64];
			sprintf(varName, "__temp_%d", i);
			image->SetVariable(i, varName);
		}
	}
		
	return image;
}


void EncodeFragment(RCX_Image *image, Fragment *f, VarPool &varPool, bool temps, int index, const RCX_Target *target)
{
	Bytecode *b = new Bytecode(varPool, temps, target);

	f->Emit(*b);
	image->SetFragment(index++, f->GetType(), f->GetNumber(),
		b->GetData(), b->GetLength(), f->GetName()->GetKey());

	delete b;
}


bool Program::Check(const RCX_Target *target, VarPool &varPool)
{
	bool ok = true;

	// find main
	if (!fMainAdded)
	{
		Error(kErr_UndefinedMain).Raise(0);
		ok = false;
	}
	else
	{
		BlockStmt *mainBlock = fTasks.GetHead()->GetBody();
		
		// insert call for init routine into main task
		if (fInitName)
		{
			Function *f = gProgram->GetFunction(fInitName);
			if (f)
			{
				InlineStmt *s = new InlineStmt();
				s->Add(f->GetBody()->Clone(0));
				mainBlock->Prepend(s);
			}
			else
			{
				Error(kErr_UnknownInit, fInitName->GetKey()).Raise(0);
				ok = false;
			}
		}
		
		// insert init stmts into main task
		mainBlock->Prepend(fInitStmts);
		fInitStmts = 0;
	}
	
	
	if (fCounts[kRCX_TaskFragment] > target->fMaxTasks)
	{
		Error(kErr_TooManyTasks, target->fMaxTasks).Raise(0);
		ok = false;
	}

	if (fCounts[kRCX_SubFragment] > target->fMaxSubs)
	{
		Error(kErr_TooManySubs, target->fMaxSubs).Raise(0);
		ok = false;
	}
	
	Fragment *f;
	for(f=fTasks.GetHead(); f; f=f->GetNext())
	{
		if (!f->Check(&varPool)) ok = false;
		varPool.ReserveUsed();
	}
	
	for(f=fSubs.GetHead(); f; f=f->GetNext())
	{
		if (!f->Check(0)) ok = false;
	}
		
	return ok;
}


Fragment *Program::GetTask(const Symbol *name)
{
	for(Fragment *f = fTasks.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


Fragment *Program::GetSub(const Symbol *name)
{
	for(Fragment *f = fSubs.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


Function *Program::GetFunction(const Symbol *name)
{
	for(Function *f = fFunctions.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


bool Program::Defined(const Symbol *name) const
{
	// this "const" hack is just to avoid writing duplicate
	// versions of GetTask(), GetSub(), and GetFunction() that
	// maintain const-ness
	Program *p = (Program*)(this);

	if (p->GetTask(name)) return true;
	if (p->GetSub(name)) return true;
	if (p->GetFunction(name)) return true;
	
	return fScopes.GetTail()->Contains(name);
}


Scope* Program::PushScope()
{
	Scope *s = new Scope(fContext);
	fScopes.InsertHead(s);
	return s;
}


void Program::PopScope()
{
	delete fScopes.RemoveHead();
}
