/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include "Bytecode.h"
#include "RCX_Cmd.h"
#include "VarPool.h"

Bytecode::Bytecode(VarPool &varPool, bool tempsAllowed, const RCX_Target *target) :
	fVarPool(varPool),
	fTempsAllowed(tempsAllowed),
	fTarget(target)
{
	fData.SetGrowStep(256);
	fLabels.SetGrowStep(50);
	fReturnLabel = NewLabel();
}


Bytecode::~Bytecode()
{
	Fixup *f;
	LoopContext *c;

	while((f=fFixups.RemoveHead()) != nil)
		delete f;

	while((c=fLoopContexts.RemoveHead()) != nil)
		delete c;
}


void Bytecode::Add(const UByte *data, int count)
{
	int n = fData.GetLength();
	
	fData.SetLength(n + count);
	memcpy(fData.GetContents() + n, data, (size_t) count);
}


void Bytecode::Add(const RCX_Cmd &cmd)
{
	int count = cmd.GetLength();
	int n = fData.GetLength();
	
	fData.SetLength(n + count);
	memcpy(fData.GetContents() + n, cmd.GetBody(), (size_t)count);
}



void Bytecode::AddJump(int label)
{
	RCX_Cmd cmd;
	
	cmd.MakeJump(0);
	Add(cmd);
	AddFixup(kPatch_SignBit, fData.GetLength() - 2, label);
}


void Bytecode::AddTest(RCX_Value v1, RCX_Relation rel, RCX_Value v2, int label)
{
	RCX_Cmd cmd;
	
	cmd.MakeTest(v1, rel, v2, 0);
	Add(cmd);
	AddFixup(kPatch_Normal, GetLength() - 2, label); 
}


void Bytecode::AddMove(int dst, RCX_Value ea)
{
	if (ea == RCX_VALUE(kRCX_VariableType, dst)) return;
	
	RCX_Cmd cmd;
	cmd.MakeVar(kRCX_SetVar, (UByte)dst, ea);
	Add(cmd);
}


int Bytecode::NewLabel()
{
	int label = fLabels.GetLength();
	
	fLabels.SetLength(label+1);
	
	return label;
}


void Bytecode::SetLabel(int label, int target)
{
	if (target == kCurrentPosition)
		fLabels[label] = GetLength();
	else
		fLabels[label] = (UShort)target;
}


void Bytecode::AddFixup(int type, UShort location, int label)
{
	Fixup *f;
	
	f = new Fixup();
	
	f->fType = type;
	f->fLocation = location;
	f->fLabel = label;
	
	fFixups.InsertHead(f);
}


void Bytecode::ApplyFixups()
{
	Fixup *f;
	
	while((f = fFixups.RemoveHead()) != nil)
	{
		int offset = fLabels[f->fLabel] - f->fLocation;

		if (f->fType == kPatch_SignBit)
		{
			UByte neg = 0;

			if (offset < 0)
			{
				neg = 0x80;
				offset = -offset;
			}

			fData[f->fLocation] = (UByte)(neg | (offset & 0x7f));
			fData[f->fLocation+1] = (UByte)((offset >> 7) & 0xff);
		}
		else
		{
			fData[f->fLocation] = (UByte)(offset & 0xff);
			fData[f->fLocation+1] = (UByte)((offset >> 8) & 0xff);
		}
		
		delete f;
	}
}


void Bytecode::PushLoopContext(bool continueAllowed)
{
	LoopContext *lc;
	
	lc = new LoopContext();
	lc->fContinueLabel = continueAllowed ? NewLabel() : kIllegalLabel;
	lc->fBreakLabel = NewLabel();
	fLoopContexts.InsertHead(lc);
}


void Bytecode::PopLoopContext(int continueTarget, int breakTarget)
{
	LoopContext *lc;
	
	lc = fLoopContexts.RemoveHead();
	if (!lc) return;	// should never happen!
	
	// its possible that fContinueLabel was illegal (as in a switch statement)
	if (lc->fContinueLabel != kIllegalLabel)
		SetLabel(lc->fContinueLabel, continueTarget);

	SetLabel(lc->fBreakLabel, breakTarget);

	delete lc;
}


int Bytecode::GetContinueLabel() const
{
	LoopContext *lc = fLoopContexts.GetHead();

	// case when no enclosing loop context is found
	if (!lc) return kIllegalLabel;
	
	return lc->fContinueLabel;
}


int Bytecode::GetBreakLabel() const
{
	LoopContext *lc = fLoopContexts.GetHead();

	// case when no enclosing loop context is found
	if (!lc) return kIllegalLabel;
	
	return lc->fBreakLabel;
}


int Bytecode::GetTempVar()
{
	if (!fTempsAllowed) return kIllegalVar;
	
	return fVarPool.AllocateTemp();
}


void Bytecode::ReleaseTempEA(RCX_Value ea)
{
	if (RCX_VALUE_TYPE(ea) != kRCX_VariableType) return;
	int v = RCX_VALUE_DATA(ea);
	
	fVarPool.ReleaseTemp(v);
}


bool Bytecode::IsTempEA(RCX_Value ea)
{
	if (RCX_VALUE_TYPE(ea) != kRCX_VariableType) return false;
	int v = RCX_VALUE_DATA(ea);
	return fVarPool.IsTemp(v);
}
